<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Atendimento;
use App\Models\Lead;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AtendimentoController extends Controller
{
    /**
     * Kanban Geral de Atendimentos - Visão Admin
     */
    public function index(Request $request)
    {
        // Filtros
        $filtroBusca = $request->input('busca', '');
        $filtroAtribuido = $request->input('atribuido', '');

        // Query base
        $query = DB::table('leads as l')
            ->leftJoin('usuarios as esp', 'l.especialista_id', '=', 'esp.id')
            ->leftJoin('equipes as eq', 'l.equipe_id', '=', 'eq.id')
            ->select(
                'l.id',
                'l.nome',
                'l.email',
                'l.telefone',
                'l.profissao',
                'l.etapa',
                'l.status',
                'l.especialista_id',
                'l.equipe_id',
                'l.data_cadastro',
                'l.data_distribuicao',
                'l.data_atualizacao',
                'l.observacoes',
                'l.horario_preferencia',
                'l.area_interesse',
                'l.data_agendamento',
                'l.horario_agendamento',
                'esp.nome as especialista_nome',
                'eq.nome as equipe_nome',
                DB::raw('CASE WHEN l.especialista_id IS NOT NULL THEN 1 ELSE 0 END as distribuido'),
                DB::raw("CASE 
                    WHEN l.etapa = 'novo' THEN 'aguardando'
                    WHEN l.etapa = 'contato' THEN 'contatado'
                    WHEN l.etapa = 'demo' THEN 'negociacao'
                    WHEN l.etapa = 'fechamento' THEN 'aprovada'
                    WHEN l.etapa = 'perdido' THEN 'rejeitada'
                    ELSE 'aguardando'
                END as status_kanban")
            );

        // Aplicar filtros
        if (!empty($filtroBusca)) {
            $query->where(function($q) use ($filtroBusca) {
                $q->where('l.nome', 'like', "%{$filtroBusca}%")
                  ->orWhere('l.email', 'like', "%{$filtroBusca}%")
                  ->orWhere('l.telefone', 'like', "%{$filtroBusca}%");
            });
        }

        if (!empty($filtroAtribuido)) {
            if ($filtroAtribuido === 'nao_atribuido') {
                $query->whereNull('l.especialista_id')->whereNull('l.equipe_id');
            } elseif (strpos($filtroAtribuido, 'esp_') === 0) {
                $espId = substr($filtroAtribuido, 4);
                $query->where('l.especialista_id', $espId);
            } elseif (strpos($filtroAtribuido, 'eq_') === 0) {
                $eqId = substr($filtroAtribuido, 3);
                $query->where('l.equipe_id', $eqId);
            }
        }

        $leads = $query->orderByRaw("
            CASE 
                WHEN l.etapa = 'novo' THEN 1
                WHEN l.etapa = 'contato' THEN 2
                WHEN l.etapa = 'demo' THEN 3
                WHEN l.etapa = 'fechamento' THEN 4
                ELSE 5
            END
        ")
        ->orderBy('l.data_atualizacao', 'desc')
        ->orderBy('l.data_cadastro', 'desc')
        ->get();

        // Especialistas para filtro
        $especialistas = DB::table('usuarios as u')
            ->leftJoin('leads as l', 'u.id', '=', 'l.especialista_id')
            ->where('u.role', 'especialista')
            ->where('u.status', 'ativo')
            ->select('u.id', 'u.nome', DB::raw('COUNT(l.id) as total_leads'))
            ->groupBy('u.id', 'u.nome')
            ->orderBy('u.nome')
            ->get();

        // Equipes para filtro
        $equipes = DB::table('equipes as eq')
            ->leftJoin('leads as l', 'eq.id', '=', 'l.equipe_id')
            ->where('eq.status', 'ativo')
            ->select('eq.id', 'eq.nome', DB::raw('COUNT(l.id) as total_leads'))
            ->groupBy('eq.id', 'eq.nome')
            ->orderBy('eq.nome')
            ->get();

        // Estatísticas gerais
        $stats = DB::table('leads')
            ->select(
                DB::raw('COUNT(*) as total'),
                DB::raw('SUM(CASE WHEN etapa = "novo" THEN 1 ELSE 0 END) as aguardando'),
                DB::raw('SUM(CASE WHEN etapa = "contato" THEN 1 ELSE 0 END) as contatado'),
                DB::raw('SUM(CASE WHEN etapa = "demo" THEN 1 ELSE 0 END) as negociacao'),
                DB::raw('SUM(CASE WHEN etapa = "fechamento" THEN 1 ELSE 0 END) as aprovada'),
                DB::raw('SUM(CASE WHEN etapa = "perdido" THEN 1 ELSE 0 END) as rejeitada'),
                DB::raw('SUM(CASE WHEN especialista_id IS NOT NULL OR equipe_id IS NOT NULL THEN 1 ELSE 0 END) as distribuidos')
            )
            ->first();

        return view('admin.atendimentos.index', compact('leads', 'especialistas', 'equipes', 'filtroBusca', 'filtroAtribuido', 'stats'));
    }

    /**
     * Atualizar status do lead via AJAX (Drag and Drop)
     */
    public function atualizarStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:aguardando,contatado,negociacao,aprovada,rejeitada',
        ]);

        $lead = DB::table('leads')->where('id', $id)->first();
        
        if (!$lead) {
            return response()->json([
                'success' => false,
                'message' => 'Lead não encontrado',
            ], 404);
        }

        // Mapear status do kanban para etapa do banco
        $etapaMap = [
            'aguardando' => 'novo',
            'contatado' => 'contato',
            'negociacao' => 'demo',
            'aprovada' => 'fechamento',
            'rejeitada' => 'perdido',
        ];

        $etapa = $etapaMap[$request->status] ?? 'novo';

        DB::table('leads')
            ->where('id', $id)
            ->update([
                'etapa' => $etapa,
                'data_atualizacao' => now(),
            ]);

        return response()->json([
            'success' => true,
            'message' => 'Status atualizado!',
            'lead_nome' => $lead->nome,
        ]);
    }
}

