<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Comissao;
use App\Models\Negociacao;
use App\Models\Usuario;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ComissaoController extends Controller
{
    /**
     * Lista todas as comissões
     */
    public function index(Request $request)
    {
        $query = DB::table('comissoes as c')
            ->join('negociacoes as n', 'c.negociacao_id', '=', 'n.id')
            ->join('usuarios as e', 'c.especialista_id', '=', 'e.id')
            ->leftJoin('leads as l', 'n.lead_id', '=', 'l.id')
            ->select(
                'c.*',
                'n.valor_total as negociacao_valor',
                'n.status as negociacao_status',
                'e.nome as especialista_nome',
                'e.email as especialista_email',
                'l.nome as lead_nome'
            );

        // Filtros
        if ($request->filled('especialista_id')) {
            $query->where('c.especialista_id', $request->especialista_id);
        }

        if ($request->filled('status')) {
            $query->where('c.status', $request->status);
        }

        if ($request->filled('data_inicio')) {
            $query->where('c.data_prevista', '>=', $request->data_inicio);
        }

        if ($request->filled('data_fim')) {
            $query->where('c.data_prevista', '<=', $request->data_fim);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('e.nome', 'like', "%{$search}%")
                  ->orWhere('l.nome', 'like', "%{$search}%")
                  ->orWhere('c.id', 'like', "%{$search}%");
            });
        }

        $comissoes = $query->orderBy('c.data_prevista', 'desc')
            ->orderBy('c.id', 'desc')
            ->paginate(50);

        // Estatísticas
        $stats = [
            'total' => DB::table('comissoes')->sum('valor'),
            'pendentes' => DB::table('comissoes')->where('status', 'pendente')->sum('valor'),
            'liberadas' => DB::table('comissoes')->where('status', 'liberada')->sum('valor'),
            'pagas' => DB::table('comissoes')->where('status', 'paga')->sum('valor'),
        ];

        // Especialistas para filtro
        $especialistas = DB::table('usuarios as u')
            ->join('usuario_categoria as uc', 'u.id', '=', 'uc.usuario_id')
            ->join('categorias_usuario as c', 'uc.categoria_id', '=', 'c.id')
            ->where('c.slug', 'especialista')
            ->where('u.status', 'ativo')
            ->where('uc.ativo', 1)
            ->select('u.id', 'u.nome')
            ->orderBy('u.nome')
            ->get();

        return view('admin.comissoes.index', compact('comissoes', 'stats', 'especialistas'));
    }

    /**
     * Exibe formulário de criação
     */
    public function create()
    {
        // Negociações fechadas sem comissão
        $negociacoes = DB::table('negociacoes as n')
            ->leftJoin('comissoes as c', 'n.id', '=', 'c.negociacao_id')
            ->join('leads as l', 'n.lead_id', '=', 'l.id')
            ->join('usuarios as e', 'n.especialista_id', '=', 'e.id')
            ->where('n.status', 'fechado')
            ->whereNull('c.id')
            ->select(
                'n.id',
                'n.valor_total',
                'n.especialista_id',
                'l.nome as lead_nome',
                'e.nome as especialista_nome'
            )
            ->orderBy('n.data_criacao', 'desc')
            ->get();

        // Especialistas
        $especialistas = DB::table('usuarios as u')
            ->join('usuario_categoria as uc', 'u.id', '=', 'uc.usuario_id')
            ->join('categorias_usuario as c', 'uc.categoria_id', '=', 'c.id')
            ->where('c.slug', 'especialista')
            ->where('u.status', 'ativo')
            ->where('uc.ativo', 1)
            ->select('u.id', 'u.nome')
            ->orderBy('u.nome')
            ->get();

        return view('admin.comissoes.create', compact('negociacoes', 'especialistas'));
    }

    /**
     * Cria uma nova comissão
     */
    public function store(Request $request)
    {
        $request->validate([
            'negociacao_id' => 'required|exists:negociacoes,id',
            'especialista_id' => 'required|exists:usuarios,id',
            'percentual' => 'required|numeric|min:0|max:100',
            'valor' => 'required|numeric|min:0',
            'data_prevista' => 'required|date',
            'status' => 'required|in:pendente,liberada,paga',
        ], [
            'negociacao_id.required' => 'Selecione uma negociação.',
            'negociacao_id.exists' => 'Negociação não encontrada.',
            'especialista_id.required' => 'Selecione um especialista.',
            'especialista_id.exists' => 'Especialista não encontrado.',
            'percentual.required' => 'Informe o percentual da comissão.',
            'percentual.numeric' => 'O percentual deve ser um número.',
            'percentual.min' => 'O percentual não pode ser negativo.',
            'percentual.max' => 'O percentual não pode ser maior que 100%.',
            'valor.required' => 'Informe o valor da comissão.',
            'valor.numeric' => 'O valor deve ser um número.',
            'valor.min' => 'O valor não pode ser negativo.',
            'data_prevista.required' => 'Informe a data prevista para pagamento.',
            'data_prevista.date' => 'Data inválida.',
            'status.required' => 'Selecione o status da comissão.',
        ]);

        // Verificar se já existe comissão para esta negociação
        $comissaoExistente = DB::table('comissoes')
            ->where('negociacao_id', $request->negociacao_id)
            ->first();

        if ($comissaoExistente) {
            return back()->withErrors(['negociacao_id' => 'Já existe uma comissão cadastrada para esta negociação.'])->withInput();
        }

        try {
            DB::table('comissoes')->insert([
                'negociacao_id' => $request->negociacao_id,
                'especialista_id' => $request->especialista_id,
                'percentual' => $request->percentual,
                'valor' => $request->valor,
                'status' => $request->status,
                'data_prevista' => $request->data_prevista,
            ]);

            return redirect()->route('admin.comissoes.index')
                ->with('success', 'Comissão criada com sucesso!');

        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Erro ao criar comissão: ' . $e->getMessage()])->withInput();
        }
    }

    /**
     * Exibe detalhes de uma comissão
     */
    public function show($id)
    {
        $comissao = DB::table('comissoes as c')
            ->join('negociacoes as n', 'c.negociacao_id', '=', 'n.id')
            ->join('usuarios as e', 'c.especialista_id', '=', 'e.id')
            ->leftJoin('leads as l', 'n.lead_id', '=', 'l.id')
            ->where('c.id', $id)
            ->select(
                'c.*',
                'n.valor_total as negociacao_valor',
                'n.status as negociacao_status',
                'n.data_criacao as negociacao_data',
                'e.nome as especialista_nome',
                'e.email as especialista_email',
                'l.nome as lead_nome',
                'l.email as lead_email',
                'l.telefone as lead_telefone'
            )
            ->first();

        if (!$comissao) {
            abort(404);
        }

        return view('admin.comissoes.show', compact('comissao'));
    }

    /**
     * Exibe formulário de edição
     */
    public function edit($id)
    {
        $comissao = DB::table('comissoes')->where('id', $id)->first();

        if (!$comissao) {
            abort(404);
        }

        // Especialistas
        $especialistas = DB::table('usuarios as u')
            ->join('usuario_categoria as uc', 'u.id', '=', 'uc.usuario_id')
            ->join('categorias_usuario as c', 'uc.categoria_id', '=', 'c.id')
            ->where('c.slug', 'especialista')
            ->where('u.status', 'ativo')
            ->where('uc.ativo', 1)
            ->select('u.id', 'u.nome')
            ->orderBy('u.nome')
            ->get();

        return view('admin.comissoes.edit', compact('comissao', 'especialistas'));
    }

    /**
     * Atualiza uma comissão
     */
    public function update(Request $request, $id)
    {
        $comissao = DB::table('comissoes')->where('id', $id)->first();

        if (!$comissao) {
            abort(404);
        }

        $request->validate([
            'especialista_id' => 'required|exists:usuarios,id',
            'percentual' => 'required|numeric|min:0|max:100',
            'valor' => 'required|numeric|min:0',
            'data_prevista' => 'required|date',
            'status' => 'required|in:pendente,liberada,paga',
        ], [
            'especialista_id.required' => 'Selecione um especialista.',
            'especialista_id.exists' => 'Especialista não encontrado.',
            'percentual.required' => 'Informe o percentual da comissão.',
            'percentual.numeric' => 'O percentual deve ser um número.',
            'percentual.min' => 'O percentual não pode ser negativo.',
            'percentual.max' => 'O percentual não pode ser maior que 100%.',
            'valor.required' => 'Informe o valor da comissão.',
            'valor.numeric' => 'O valor deve ser um número.',
            'valor.min' => 'O valor não pode ser negativo.',
            'data_prevista.required' => 'Informe a data prevista para pagamento.',
            'data_prevista.date' => 'Data inválida.',
            'status.required' => 'Selecione o status da comissão.',
        ]);

        try {
            DB::table('comissoes')
                ->where('id', $id)
                ->update([
                    'especialista_id' => $request->especialista_id,
                    'percentual' => $request->percentual,
                    'valor' => $request->valor,
                    'status' => $request->status,
                    'data_prevista' => $request->data_prevista,
                ]);

            return redirect()->route('admin.comissoes.index')
                ->with('success', 'Comissão atualizada com sucesso!');

        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Erro ao atualizar comissão: ' . $e->getMessage()])->withInput();
        }
    }

    /**
     * Remove uma comissão
     */
    public function destroy($id)
    {
        $comissao = DB::table('comissoes')->where('id', $id)->first();

        if (!$comissao) {
            abort(404);
        }

        // Não permitir deletar comissões pagas
        if ($comissao->status === 'paga') {
            return back()->withErrors(['error' => 'Não é possível excluir uma comissão já paga.']);
        }

        try {
            DB::table('comissoes')->where('id', $id)->delete();

            return redirect()->route('admin.comissoes.index')
                ->with('success', 'Comissão excluída com sucesso!');

        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Erro ao excluir comissão: ' . $e->getMessage()]);
        }
    }

    /**
     * Atualiza o status de uma comissão (AJAX)
     */
    public function toggleStatus(Request $request, $id)
    {
        $comissao = DB::table('comissoes')->where('id', $id)->first();

        if (!$comissao) {
            return response()->json([
                'success' => false,
                'message' => 'Comissão não encontrada'
            ], 404);
        }

        $novoStatus = $request->input('status');

        if (!in_array($novoStatus, ['pendente', 'liberada', 'paga'])) {
            return response()->json([
                'success' => false,
                'message' => 'Status inválido'
            ], 400);
        }

        try {
            DB::table('comissoes')
                ->where('id', $id)
                ->update(['status' => $novoStatus]);

            return response()->json([
                'success' => true,
                'message' => 'Status atualizado com sucesso!',
                'novo_status' => $novoStatus
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao atualizar status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Calcula comissão automaticamente baseada na negociação
     */
    public function calcularComissao(Request $request)
    {
        $request->validate([
            'negociacao_id' => 'required|exists:negociacoes,id',
            'percentual' => 'required|numeric|min:0|max:100',
        ]);

        $negociacao = DB::table('negociacoes')
            ->where('id', $request->negociacao_id)
            ->first();

        if (!$negociacao) {
            return response()->json([
                'success' => false,
                'message' => 'Negociação não encontrada'
            ], 404);
        }

        $valorComissao = ($negociacao->valor_total * $request->percentual) / 100;

        return response()->json([
            'success' => true,
            'valor' => number_format($valorComissao, 2, '.', ''),
            'valor_total' => $negociacao->valor_total
        ]);
    }
}

