<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Contrato;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class ContratoController extends Controller
{
    public function index()
    {
        $contratos = Contrato::with('criador')
            ->orderBy('data_upload', 'desc')
            ->paginate(20);

        return view('admin.contratos.index', compact('contratos'));
    }

    public function create()
    {
        return view('admin.contratos.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'tipo' => 'required|in:regular,bolsista',
            'arquivo' => 'required|file|mimes:pdf|max:20480', // 20MB
        ]);

        $file = $request->file('arquivo');
        $filename = 'contrato_' . $request->tipo . '_' . time() . '.pdf';
        
        // Salvar arquivo
        $path = $file->storeAs('contratos', $filename, 'public');

        // Desativar contratos anteriores do mesmo tipo
        Contrato::where('tipo', $request->tipo)
            ->where('ativo', 1)
            ->update(['ativo' => 0]);

        Contrato::create([
            'tipo' => $request->tipo,
            'arquivo' => $filename,
            'ativo' => 1,
            'upload_por' => Auth::id(),
            'data_upload' => now(),
            'ano' => date('Y'),
        ]);

        return redirect()->route('admin.contratos.index')
            ->with('success', 'Contrato enviado com sucesso!');
    }

    public function show($id)
    {
        $contrato = Contrato::with('criador')->findOrFail($id);
        return view('admin.contratos.show', compact('contrato'));
    }

    public function edit($id)
    {
        $contrato = Contrato::findOrFail($id);
        return view('admin.contratos.edit', compact('contrato'));
    }

    public function update(Request $request, $id)
    {
        $contrato = Contrato::findOrFail($id);

        $request->validate([
            'tipo' => 'required|in:regular,bolsista',
            'ativo' => 'required|boolean',
        ]);

        // Se está ativando, desativar outros do mesmo tipo
        if ($request->ativo == 1) {
            Contrato::where('tipo', $request->tipo)
                ->where('id', '!=', $id)
                ->update(['ativo' => 0]);
        }

        $contrato->update([
            'tipo' => $request->tipo,
            'ativo' => $request->ativo,
        ]);

        return redirect()->route('admin.contratos.index')
            ->with('success', 'Contrato atualizado com sucesso!');
    }

    public function destroy($id)
    {
        $contrato = Contrato::findOrFail($id);

        // Desativar ao invés de deletar
        $contrato->update(['ativo' => 0]);

        return redirect()->route('admin.contratos.index')
            ->with('success', 'Contrato desativado com sucesso!');
    }

    public function toggleStatus($id)
    {
        $contrato = Contrato::findOrFail($id);

        // Se está ativando, desativar outros do mesmo tipo
        if (!$contrato->ativo) {
            Contrato::where('tipo', $contrato->tipo)
                ->where('id', '!=', $id)
                ->update(['ativo' => 0]);
        }

        $contrato->update(['ativo' => !$contrato->ativo]);

        return response()->json([
            'success' => true,
            'message' => 'Status atualizado!',
            'novo_status' => $contrato->ativo,
        ]);
    }
}

