<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Lead;
use App\Models\Venda;
use App\Models\Especialista;
use App\Models\Campanha;
use App\Models\Voucher;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    /**
     * Exibe o dashboard administrativo
     */
    public function index(Request $request)
    {
        // Parâmetros de período
        $start = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $end = $request->input('end_date', Carbon::now()->format('Y-m-d'));

        // Período anterior (MoM - Month over Month)
        $startAnterior = Carbon::parse($start)->subMonth()->format('Y-m-d');
        $endAnterior = Carbon::parse($end)->subMonth()->format('Y-m-d');

        // Ano anterior (YoY - Year over Year)
        $startAnoAnterior = Carbon::parse($start)->subYear()->format('Y-m-d');
        $endAnoAnterior = Carbon::parse($end)->subYear()->format('Y-m-d');

        // ============================================
        // KPI 1: LEADS CAPTADOS
        // ============================================
        $leadsAtual = Lead::whereBetween(DB::raw('DATE(data_cadastro)'), [$start, $end])->count();
        $leadsAnterior = Lead::whereBetween(DB::raw('DATE(data_cadastro)'), [$startAnterior, $endAnterior])->count();
        $leadsAnoAnterior = Lead::whereBetween(DB::raw('DATE(data_cadastro)'), [$startAnoAnterior, $endAnoAnterior])->count();

        $varLeadsMensal = $leadsAnterior > 0 ? (($leadsAtual - $leadsAnterior) / $leadsAnterior) * 100 : 0;
        $varLeadsAnual = $leadsAnoAnterior > 0 ? (($leadsAtual - $leadsAnoAnterior) / $leadsAnoAnterior) * 100 : 0;

        // ============================================
        // KPI 2: MATRÍCULAS REALIZADAS
        // ============================================
        $vendasAtual = DB::table('matriculas')
            ->whereBetween(DB::raw('DATE(data_matricula)'), [$start, $end])
            ->where('status', 'ativa')
            ->count();
        $vendasAnterior = DB::table('matriculas')
            ->whereBetween(DB::raw('DATE(data_matricula)'), [$startAnterior, $endAnterior])
            ->where('status', 'ativa')
            ->count();
        $vendasAnoAnterior = DB::table('matriculas')
            ->whereBetween(DB::raw('DATE(data_matricula)'), [$startAnoAnterior, $endAnoAnterior])
            ->where('status', 'ativa')
            ->count();

        $varVendasMensal = $vendasAnterior > 0 ? (($vendasAtual - $vendasAnterior) / $vendasAnterior) * 100 : 0;
        $varVendasAnual = $vendasAnoAnterior > 0 ? (($vendasAtual - $vendasAnoAnterior) / $vendasAnoAnterior) * 100 : 0;

        // ============================================
        // KPI 3: RECEITA (Negociações)
        // ============================================
        $receitaAtual = DB::table('negociacoes')
            ->whereBetween(DB::raw('DATE(data_criacao)'), [$start, $end])
            ->whereIn('status', ['aprovada', 'finalizada'])
            ->sum('valor_total');
        $receitaAnterior = DB::table('negociacoes')
            ->whereBetween(DB::raw('DATE(data_criacao)'), [$startAnterior, $endAnterior])
            ->whereIn('status', ['aprovada', 'finalizada'])
            ->sum('valor_total');
        $receitaAnoAnterior = DB::table('negociacoes')
            ->whereBetween(DB::raw('DATE(data_criacao)'), [$startAnoAnterior, $endAnoAnterior])
            ->whereIn('status', ['aprovada', 'finalizada'])
            ->sum('valor_total');

        $varReceitaMensal = $receitaAnterior > 0 ? (($receitaAtual - $receitaAnterior) / $receitaAnterior) * 100 : 0;
        $varReceitaAnual = $receitaAnoAnterior > 0 ? (($receitaAtual - $receitaAnoAnterior) / $receitaAnoAnterior) * 100 : 0;

        // ============================================
        // KPI 4: TAXA DE CONVERSÃO
        // ============================================
        $taxaConversaoAtual = $leadsAtual > 0 ? ($vendasAtual / $leadsAtual) * 100 : 0;
        $taxaConversaoAnterior = $leadsAnterior > 0 ? ($vendasAnterior / $leadsAnterior) * 100 : 0;

        // ============================================
        // DADOS COMPLEMENTARES
        // ============================================

        // Campanhas de Marketing (simplificado)
        $campanhas = DB::table('links_marketing')
            ->whereBetween('data_criacao', [$start, $end])
            ->get();

        // Estatísticas de Vouchers
        $vouchersAtivos = DB::table('vouchers')->where('ativo', 1)->count();
        $vouchersUsados = DB::table('vouchers')
            ->where('quantidade_usada', '>', 0)
            ->count();

        // Performance dos Especialistas (simplificado)
        $especialistas = DB::table('usuarios')
            ->where('role', 'especialista')
            ->where('status', 'ativo')
            ->select('id', 'nome')
            ->limit(10)
            ->get()
            ->map(function($esp) use ($start, $end) {
                // Contar leads
                $leads_ativos = DB::table('leads')
                    ->where('especialista_id', $esp->id)
                    ->whereBetween(DB::raw('DATE(data_cadastro)'), [$start, $end])
                    ->count();
                
                // Contar matrículas (através de alunos)
                $vendas_realizadas = DB::table('matriculas as m')
                    ->join('alunos as a', 'm.aluno_id', '=', 'a.id')
                    ->where('a.especialista_id', $esp->id)
                    ->whereBetween(DB::raw('DATE(m.data_matricula)'), [$start, $end])
                    ->where('m.status', 'ativa')
                    ->count();
                
                // Somar receita
                $receita_gerada = DB::table('negociacoes')
                    ->where('especialista_id', $esp->id)
                    ->whereBetween(DB::raw('DATE(data_criacao)'), [$start, $end])
                    ->whereIn('status', ['aprovada', 'finalizada'])
                    ->sum('valor_total') ?? 0;
                
                return (object)[
                    'usuario' => (object)['nome' => $esp->nome],
                    'leads_ativos' => $leads_ativos,
                    'vendas_realizadas' => $vendas_realizadas,
                    'receita_gerada' => $receita_gerada,
                ];
            });

        // Evolução diária de leads
        $evolucaoDiaria = Lead::selectRaw('DATE(data_cadastro) as data, COUNT(*) as total')
            ->whereBetween(DB::raw('DATE(data_cadastro)'), [$start, $end])
            ->groupBy('data')
            ->orderBy('data')
            ->get();

        // Distribuição por status
        $dadosStatus = Lead::selectRaw('status, COUNT(*) as total')
            ->whereBetween(DB::raw('DATE(data_cadastro)'), [$start, $end])
            ->groupBy('status')
            ->get();

        return view('admin.dashboard', compact(
            'start',
            'end',
            'leadsAtual',
            'varLeadsMensal',
            'varLeadsAnual',
            'vendasAtual',
            'varVendasMensal',
            'varVendasAnual',
            'receitaAtual',
            'varReceitaMensal',
            'varReceitaAnual',
            'taxaConversaoAtual',
            'taxaConversaoAnterior',
            'campanhas',
            'vouchersAtivos',
            'vouchersUsados',
            'especialistas',
            'evolucaoDiaria',
            'dadosStatus'
        ));
    }
}

