<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Equipe;
use App\Models\Usuario;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class EquipeController extends Controller
{
    public function index()
    {
        $equipes = Equipe::with('coordenador')
            ->orderBy('nome')
            ->paginate(20);

        // Adicionar contagem de especialistas
        foreach ($equipes as $equipe) {
            $equipe->especialistas_count = DB::table('equipe_especialistas')
                ->where('equipe_id', $equipe->id)
                ->where('status', 'ativo')
                ->count();
        }

        return view('admin.equipes.index', compact('equipes'));
    }

    public function create()
    {
        $coordenadores = Usuario::where('role', 'coordenador')
            ->where('status', 'ativo')
            ->orderBy('nome')
            ->get();

        $especialistas = Usuario::where('role', 'especialista')
            ->where('status', 'ativo')
            ->orderBy('nome')
            ->get();

        return view('admin.equipes.create', compact('coordenadores', 'especialistas'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'nome' => 'required|string|max:255|unique:equipes,nome',
            'coordenador_id' => 'required|exists:usuarios,id',
            'meta_mensal' => 'nullable|numeric|min:0',
            'status' => 'required|in:ativo,inativo',
        ]);

        $equipe = Equipe::create([
            'nome' => $request->nome,
            'coordenador_id' => $request->coordenador_id,
            'meta_mensal' => $request->meta_mensal ?? 0,
            'status' => $request->status,
        ]);

        // Vincular especialistas na tabela pivot
        if ($request->filled('especialistas')) {
            foreach ($request->especialistas as $especialistaId) {
                DB::table('equipe_especialistas')->insert([
                    'equipe_id' => $equipe->id,
                    'especialista_id' => $especialistaId,
                    'data_entrada' => now(),
                    'status' => 'ativo',
                ]);
            }
        }

        return redirect()->route('admin.equipes.index')
            ->with('success', 'Equipe criada com sucesso!');
    }

    public function show($id)
    {
        $equipe = Equipe::with('coordenador')->findOrFail($id);

        // Buscar especialistas da equipe
        $especialistas = DB::table('equipe_especialistas')
            ->join('usuarios', 'equipe_especialistas.especialista_id', '=', 'usuarios.id')
            ->where('equipe_especialistas.equipe_id', $id)
            ->where('equipe_especialistas.status', 'ativo')
            ->select('usuarios.*', 'equipe_especialistas.data_entrada')
            ->get();

        // Buscar leads da equipe
        $leads = Lead::where('equipe_id', $id)->get();

        return view('admin.equipes.show', compact('equipe', 'especialistas', 'leads'));
    }

    public function edit($id)
    {
        $equipe = Equipe::findOrFail($id);

        $coordenadores = Usuario::where('role', 'coordenador')
            ->where('status', 'ativo')
            ->orderBy('nome')
            ->get();

        // Buscar todos os especialistas
        $especialistas = Usuario::where('role', 'especialista')
            ->where('status', 'ativo')
            ->orderBy('nome')
            ->get();

        // Buscar especialistas já vinculados a esta equipe
        $especialistasVinculados = DB::table('equipe_especialistas')
            ->where('equipe_id', $id)
            ->where('status', 'ativo')
            ->pluck('especialista_id')
            ->toArray();

        return view('admin.equipes.edit', compact('equipe', 'coordenadores', 'especialistas', 'especialistasVinculados'));
    }

    public function update(Request $request, $id)
    {
        $equipe = Equipe::findOrFail($id);

        $request->validate([
            'nome' => 'required|string|max:255|unique:equipes,nome,' . $id,
            'coordenador_id' => 'required|exists:usuarios,id',
            'meta_mensal' => 'nullable|numeric|min:0',
            'status' => 'required|in:ativo,inativo',
        ]);

        $equipe->update([
            'nome' => $request->nome,
            'coordenador_id' => $request->coordenador_id,
            'meta_mensal' => $request->meta_mensal ?? 0,
            'status' => $request->status,
        ]);

        // Atualizar especialistas
        // Remover todos da equipe atual
        DB::table('equipe_especialistas')
            ->where('equipe_id', $equipe->id)
            ->update(['status' => 'inativo']);

        // Adicionar os selecionados
        if ($request->filled('especialistas')) {
            foreach ($request->especialistas as $especialistaId) {
                // Verificar se já existe
                $existe = DB::table('equipe_especialistas')
                    ->where('equipe_id', $equipe->id)
                    ->where('especialista_id', $especialistaId)
                    ->first();

                if ($existe) {
                    // Reativar
                    DB::table('equipe_especialistas')
                        ->where('equipe_id', $equipe->id)
                        ->where('especialista_id', $especialistaId)
                        ->update(['status' => 'ativo']);
                } else {
                    // Criar novo
                    DB::table('equipe_especialistas')->insert([
                        'equipe_id' => $equipe->id,
                        'especialista_id' => $especialistaId,
                        'data_entrada' => now(),
                        'status' => 'ativo',
                    ]);
                }
            }
        }

        return redirect()->route('admin.equipes.index')
            ->with('success', 'Equipe atualizada com sucesso!');
    }

    public function destroy($id)
    {
        $equipe = Equipe::findOrFail($id);

        // Verificar se tem especialistas ativos
        $temEspecialistas = DB::table('equipe_especialistas')
            ->where('equipe_id', $id)
            ->where('status', 'ativo')
            ->exists();

        if ($temEspecialistas) {
            return back()->with('error', 'Não é possível excluir uma equipe com especialistas vinculados.');
        }

        $equipe->update(['status' => 'inativo']);

        return redirect()->route('admin.equipes.index')
            ->with('success', 'Equipe desativada com sucesso!');
    }
}

