<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class ValorCursoController extends Controller
{
    public function index()
    {
        // Buscar valor integral ativo
        $valorAtual = DB::table('valores_curso as v')
            ->leftJoin('usuarios as u', 'v.criado_por', '=', 'u.id')
            ->select('v.*', 'u.nome as criado_por_nome')
            ->where('v.ativo', 1)
            ->orderBy('v.data_criacao', 'desc')
            ->first();

        // Se não houver valor ativo, criar um padrão
        if (!$valorAtual) {
            DB::table('valores_curso')->insert([
                'valor_integral' => 825.00,
                'ativo' => 1,
                'criado_por' => auth()->id(),
                'data_criacao' => now(),
            ]);

            $valorAtual = DB::table('valores_curso')
                ->where('ativo', 1)
                ->first();
        }

        // Buscar histórico de valores
        $historicoValores = DB::table('valores_curso as v')
            ->leftJoin('usuarios as u', 'v.criado_por', '=', 'u.id')
            ->select('v.*', 'u.nome as criado_por_nome')
            ->orderBy('v.data_criacao', 'desc')
            ->limit(20)
            ->get();

        // Estatísticas
        $totalAlteracoes = $historicoValores->count();
        $valorIntegralAtual = $valorAtual->valor_integral ?? 825.00;
        $valorComDesconto20 = $valorIntegralAtual * 0.80;
        $valorMinimoAprovacao = 660.00;

        return view('admin.valores-curso.index', compact(
            'valorAtual',
            'historicoValores',
            'totalAlteracoes',
            'valorIntegralAtual',
            'valorComDesconto20',
            'valorMinimoAprovacao'
        ));
    }

    public function atualizarValorIntegral(Request $request)
    {
        $request->validate([
            'valor_integral' => 'required|numeric|min:0.01',
        ], [
            'valor_integral.required' => 'O valor integral é obrigatório.',
            'valor_integral.numeric' => 'O valor deve ser numérico.',
            'valor_integral.min' => 'O valor deve ser maior que zero.',
        ]);

        DB::beginTransaction();
        try {
            // Desativar todos os valores anteriores
            DB::table('valores_curso')->update(['ativo' => 0]);

            // Inserir novo valor ativo
            DB::table('valores_curso')->insert([
                'valor_integral' => $request->valor_integral,
                'ativo' => 1,
                'criado_por' => auth()->id(),
                'data_criacao' => now(),
            ]);

            DB::commit();

            return redirect()->route('admin.valores-curso.index')
                ->with('success', 'Valor integral atualizado com sucesso para R$ ' . number_format($request->valor_integral, 2, ',', '.') . '!');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Erro ao atualizar valor: ' . $e->getMessage());
        }
    }
}


