<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Negociacao;
use App\Models\Lead;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class VendaController extends Controller
{
    /**
     * Lista todas as vendas aguardando aprovação
     */
    public function index(Request $request)
    {
        // Buscar negociações aguardando aprovação (status 'andamento' e lead com etapa 'fechamento')
        $query = DB::table('negociacoes as n')
            ->join('leads as l', 'n.lead_id', '=', 'l.id')
            ->leftJoin('usuarios as e', 'n.especialista_id', '=', 'e.id')
            ->leftJoin('usuarios as a', 'n.aprovado_por', '=', 'a.id')
            ->where('n.status', 'andamento')
            ->where('l.etapa', 'fechamento')
            ->select(
                'n.id',
                'n.lead_id',
                'n.especialista_id',
                'n.valor_proposta',
                'n.valor_total',
                'n.data_criacao',
                'n.observacoes',
                'l.nome as lead_nome',
                'l.email as lead_email',
                'l.telefone as lead_telefone',
                'e.nome as especialista_nome',
                'a.nome as aprovador_nome'
            );

        // Filtros
        if ($request->filled('especialista_id')) {
            $query->where('n.especialista_id', $request->especialista_id);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('l.nome', 'like', "%{$search}%")
                  ->orWhere('l.email', 'like', "%{$search}%")
                  ->orWhere('l.telefone', 'like', "%{$search}%");
            });
        }

        $vendas = $query->orderBy('n.data_criacao', 'desc')->paginate(50);

        // Estatísticas
        $stats = [
            'aguardando_aprovacao' => DB::table('negociacoes as n')
                ->join('leads as l', 'n.lead_id', '=', 'l.id')
                ->where('n.status', 'andamento')
                ->where('l.etapa', 'fechamento')
                ->count(),
            'aprovadas' => DB::table('negociacoes')
                ->where('status', 'fechado')
                ->count(),
            'total_mes' => DB::table('matriculas')
                ->whereMonth('data_matricula', now()->month)
                ->whereYear('data_matricula', now()->year)
                ->sum('valor_total') ?? 0,
        ];

        return view('admin.vendas.index', compact('vendas', 'stats'));
    }

    /**
     * Exibe detalhes de uma venda
     */
    public function show($id)
    {
        $venda = Negociacao::with(['lead', 'aprovador'])->findOrFail($id);

        // Buscar especialista
        $especialista = Usuario::find($venda->especialista_id);

        return view('admin.vendas.show', compact('venda', 'especialista'));
    }

    /**
     * Aprova uma venda e cria a matrícula
     */
    public function aprovar(Request $request, $id)
    {
        $request->validate([
            'curso' => 'required|string|max:150',
            'valor_total' => 'required|numeric|min:0',
        ], [
            'curso.required' => 'Informe o curso.',
            'valor_total.required' => 'Informe o valor total.',
        ]);

        $now = Carbon::now('America/Sao_Paulo');
        
        DB::beginTransaction();
        try {
            // Buscar negociação
            $negociacao = DB::table('negociacoes')
                ->where('id', $id)
                ->where('status', 'andamento')
                ->first();

            if (!$negociacao) {
                return response()->json([
                    'success' => false,
                    'message' => 'Esta negociação não está aguardando aprovação.',
                ], 400);
            }

            // Buscar lead
            $lead = DB::table('leads')->where('id', $negociacao->lead_id)->first();
            
            if (!$lead) {
                return response()->json([
                    'success' => false,
                    'message' => 'Lead não encontrado.',
                ], 404);
            }

            // Verificar se já existe aluno para este lead
            $aluno = DB::table('alunos')
                ->where('email', $lead->email)
                ->orWhere('cpf', $lead->cpf)
                ->first();

            // Se não existir aluno, criar
            if (!$aluno) {
                $alunoId = DB::table('alunos')->insertGetId([
                    'nome' => $lead->nome,
                    'email' => $lead->email,
                    'telefone' => $lead->telefone,
                    'cpf' => $lead->cpf ?? null,
                    'status' => 'ativo',
                    'especialista_id' => $negociacao->especialista_id,
                    'data_cadastro' => $now,
                    'tipo_contrato' => 'regular',
                ]);
            } else {
                $alunoId = $aluno->id;
            }

            // Criar vínculo entre lead e aluno
            $vinculoExistente = DB::table('vinculo')
                ->where('lead_id', $lead->id)
                ->where('aluno_id', $alunoId)
                ->first();

            if (!$vinculoExistente) {
                DB::table('vinculo')->insert([
                    'lead_id' => $lead->id,
                    'aluno_id' => $alunoId,
                    'especialista_id' => $negociacao->especialista_id,
                    'status' => 'ativo',
                    'data_criacao' => $now,
                ]);
            }

            // Criar matrícula
            $matriculaId = DB::table('matriculas')->insertGetId([
                'aluno_id' => $alunoId,
                'especialista_id' => $negociacao->especialista_id,
                'curso' => $request->curso,
                'valor_total' => $request->valor_total,
                'status' => 'ativa',
                'data_matricula' => $now->format('Y-m-d'),
            ]);

            // Atualizar negociação
            DB::table('negociacoes')
                ->where('id', $id)
                ->update([
                    'status' => 'fechado',
                    'aluno_id' => $alunoId,
                    'valor_total' => $request->valor_total,
                    'data_fechamento' => $now,
                    'aprovado_por' => Auth::id(),
                ]);

            // Atualizar lead
            DB::table('leads')
                ->where('id', $lead->id)
                ->update([
                    'status' => 'convertido',
                    'etapa' => 'convertido',
                ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Negociação aprovada e matrícula criada com sucesso!',
                'matricula_id' => $matriculaId,
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Erro ao aprovar: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Rejeita uma venda
     */
    public function rejeitar(Request $request, $id)
    {
        $request->validate([
            'motivo' => 'required|string|max:500',
        ], [
            'motivo.required' => 'Informe o motivo da rejeição.',
        ]);

        $venda = Negociacao::findOrFail($id);

        if ($venda->status !== 'pendente') {
            return response()->json([
                'success' => false,
                'message' => 'Esta negociação não está pendente de aprovação.',
            ], 400);
        }

        $venda->update([
            'status' => 'rejeitada',
            'data_aprovacao' => now(),
            'aprovado_por' => Auth::id(),
            'observacoes' => ($venda->observacoes ? $venda->observacoes . "\n\n" : '') 
                . "REJEITADA: " . $request->motivo,
        ]);

        // Retornar lead para negociação
        if ($venda->lead) {
            $venda->lead->update(['status' => 'negociacao']);
        }

        return response()->json([
            'success' => true,
            'message' => 'Negociação rejeitada.',
        ]);
    }
}

