<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Voucher;
use App\Models\Usuario;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class VoucherController extends Controller
{
    public function index()
    {
        $vouchers = Voucher::with('criador')
            ->orderBy('data_criacao', 'desc')
            ->paginate(50);

        return view('admin.vouchers.index', compact('vouchers'));
    }

    public function create()
    {
        $especialistas = Usuario::where('role', 'especialista')
            ->where('status', 'ativo')
            ->orderBy('nome')
            ->get();

        return view('admin.vouchers.create', compact('especialistas'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'codigo' => 'required|string|max:50|unique:vouchers,codigo',
            'nome' => 'nullable|string|max:150',
            'descricao' => 'nullable|string|max:255',
            'tipo' => 'required|in:desconto,brinde',
            'percentual_desconto' => 'nullable|numeric|min:0|max:100',
            'valor_desconto' => 'nullable|numeric|min:0',
            'valor_minimo' => 'nullable|numeric|min:0',
            'quantidade_total' => 'required|integer|min:1',
            'validade' => 'nullable|date|after:today',
            'abrangencia' => 'required|in:global,individual',
            'tipo_quantidade' => 'required|in:coletiva,individual',
            'especialistas' => 'required_if:abrangencia,individual|array',
        ], [
            'especialistas.required_if' => 'Selecione pelo menos um especialista para voucher individual.',
        ]);

        DB::beginTransaction();
        try {
            $voucher = Voucher::create([
                'codigo' => strtoupper($request->codigo),
                'nome' => $request->nome,
                'descricao' => $request->descricao,
                'tipo' => $request->tipo,
                'percentual_desconto' => $request->percentual_desconto,
                'valor_desconto' => $request->valor_desconto,
                'valor_minimo' => $request->valor_minimo,
                'quantidade_total' => $request->quantidade_total,
                'quantidade_usada' => 0,
                'validade' => $request->validade,
                'ativo' => 1,
                'status' => 'ativo',
                'criado_por' => Auth::id(),
                'data_criacao' => now(),
            ]);

            // Se for individual, vincular aos especialistas selecionados
            if ($request->abrangencia === 'individual' && $request->filled('especialistas')) {
                foreach ($request->especialistas as $especialistaId) {
                    DB::table('voucher_especialista')->insert([
                        'voucher_id' => $voucher->id,
                        'especialista_id' => $especialistaId,
                    ]);
                }
            }

            DB::commit();

            $mensagem = 'Voucher criado com sucesso! ';
            $mensagem .= $request->abrangencia === 'global' ? '(Global - Todos os especialistas)' : '(Individual - ' . count($request->especialistas ?? []) . ' especialistas)';
            $mensagem .= ' | Quantidade: ' . ($request->tipo_quantidade === 'coletiva' ? 'Coletiva' : 'Individual por especialista');

            return redirect()->route('admin.vouchers.index')
                ->with('success', $mensagem);

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Erro ao criar voucher: ' . $e->getMessage())->withInput();
        }
    }

    public function show($id)
    {
        $voucher = Voucher::with('criador')->findOrFail($id);
        return view('admin.vouchers.show', compact('voucher'));
    }

    public function edit($id)
    {
        $voucher = Voucher::findOrFail($id);
        return view('admin.vouchers.edit', compact('voucher'));
    }

    public function update(Request $request, $id)
    {
        $voucher = Voucher::findOrFail($id);

        $request->validate([
            'codigo' => 'required|string|max:50|unique:vouchers,codigo,' . $id,
            'tipo' => 'required|in:percentual,valor',
            'valor_desconto' => 'required_if:tipo,valor|nullable|numeric|min:0',
            'percentual_desconto' => 'required_if:tipo,percentual|nullable|numeric|min:0|max:100',
            'data_validade' => 'nullable|date',
            'status' => 'required|in:ativo,inativo,usado',
        ]);

        $voucher->update([
            'codigo' => strtoupper($request->codigo),
            'tipo' => $request->tipo,
            'valor_desconto' => $request->tipo === 'valor' ? $request->valor_desconto : null,
            'percentual_desconto' => $request->tipo === 'percentual' ? $request->percentual_desconto : null,
            'data_validade' => $request->data_validade,
            'status' => $request->status,
        ]);

        return redirect()->route('admin.vouchers.index')
            ->with('success', 'Voucher atualizado com sucesso!');
    }

    public function destroy($id)
    {
        $voucher = Voucher::findOrFail($id);

        if ($voucher->quantidade_usada > 0) {
            return back()->with('error', 'Não é possível excluir um voucher já utilizado.');
        }

        $voucher->delete();

        return redirect()->route('admin.vouchers.index')
            ->with('success', 'Voucher excluído com sucesso!');
    }

    public function toggleStatus($id)
    {
        $voucher = Voucher::findOrFail($id);

        if ($voucher->quantidade_usada >= $voucher->quantidade_total) {
            return response()->json([
                'success' => false,
                'message' => 'Não é possível alterar o status de um voucher totalmente utilizado.',
            ], 400);
        }

        $voucher->update([
            'ativo' => !$voucher->ativo,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Status atualizado com sucesso!',
            'novo_status' => $voucher->ativo,
        ]);
    }
}

