<?php

namespace App\Http\Controllers\Coordenador;

use App\Http\Controllers\Controller;
use App\Models\Equipe;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class AlunoController extends Controller
{
    /**
     * Exibe a lista de alunos da equipe
     */
    public function index(Request $request)
    {
        $coordenador = Auth::user();
        
        // Buscar equipes do coordenador
        $equipes = Equipe::where('coordenador_id', $coordenador->id)
            ->where('status', 'ativo')
            ->get();
        
        if ($equipes->isEmpty()) {
            return redirect()->route('coordenador.dashboard')
                ->with('error', 'Você não possui equipes associadas.');
        }
        
        $equipeIds = $equipes->pluck('id');
        
        // Filtros
        $status = $request->get('status', '');
        $especialistaId = $request->get('especialista_id', '');
        $modulo = $request->get('modulo', '');
        $search = $request->get('search', '');
        
        // Estatísticas dos alunos
        $statsAlunos = DB::table('alunos as a')
            ->join('equipe_especialistas as ee', 'a.especialista_id', '=', 'ee.especialista_id')
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo')
            ->selectRaw('
                COUNT(*) as total_alunos,
                SUM(CASE WHEN a.status = "ativo" THEN 1 ELSE 0 END) as alunos_ativos,
                SUM(CASE WHEN a.data_conclusao IS NOT NULL THEN 1 ELSE 0 END) as alunos_concluidos,
                SUM(CASE WHEN a.status = "trancado" THEN 1 ELSE 0 END) as alunos_parados,
                SUM(CASE WHEN a.progresso_geral >= 80 THEN 1 ELSE 0 END) as alunos_avancados,
                SUM(CASE WHEN a.progresso_geral < 30 THEN 1 ELSE 0 END) as alunos_iniciantes,
                AVG(COALESCE(a.progresso_geral, 0)) as progresso_medio
            ')
            ->first();
        
        // Query base
        $query = DB::table('alunos as a')
            ->join('equipe_especialistas as ee', 'a.especialista_id', '=', 'ee.especialista_id')
            ->join('usuarios as u', 'a.especialista_id', '=', 'u.id')
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo');
        
        // Aplicar filtros
        if ($status) {
            if ($status === 'concluido') {
                $query->whereNotNull('a.data_conclusao');
            } else {
                $query->where('a.status', $status);
            }
        }
        
        if ($especialistaId) {
            $query->where('a.especialista_id', $especialistaId);
        }
        
        if ($modulo) {
            $query->where('a.modulo_atual', $modulo);
        }
        
        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('a.nome', 'like', "%{$search}%")
                  ->orWhere('a.email', 'like', "%{$search}%")
                  ->orWhere('a.telefone', 'like', "%{$search}%");
            });
        }
        
        // Obter alunos
        $alunos = $query->select(
                'a.*',
                'u.nome as especialista_nome',
                'u.email as especialista_email'
            )
            ->orderBy('a.data_cadastro', 'desc')
            ->get();
        
        // Especialistas para filtro
        $especialistas = DB::table('equipe_especialistas as ee')
            ->join('usuarios as u', 'ee.especialista_id', '=', 'u.id')
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo')
            ->where('u.role', 'especialista')
            ->where('u.status', 'ativo')
            ->select('u.id', 'u.nome')
            ->orderBy('u.nome')
            ->get();
        
        // Módulos disponíveis
        $modulos = DB::table('alunos as a')
            ->join('equipe_especialistas as ee', 'a.especialista_id', '=', 'ee.especialista_id')
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo')
            ->whereNotNull('a.modulo_atual')
            ->where('a.modulo_atual', '!=', '')
            ->distinct()
            ->pluck('a.modulo_atual')
            ->sort()
            ->values();
        
        // Distribuição por status
        $distribuicaoStatus = DB::table('alunos as a')
            ->join('equipe_especialistas as ee', 'a.especialista_id', '=', 'ee.especialista_id')
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo')
            ->select('a.status', DB::raw('COUNT(*) as total'))
            ->groupBy('a.status')
            ->orderByDesc('total')
            ->get();
        
        // Distribuição por módulo
        $distribuicaoModulo = DB::table('alunos as a')
            ->join('equipe_especialistas as ee', 'a.especialista_id', '=', 'ee.especialista_id')
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo')
            ->selectRaw('COALESCE(CAST(a.modulo_atual AS CHAR), "Não definido") as modulo, COUNT(*) as total')
            ->groupBy('a.modulo_atual')
            ->orderByDesc('total')
            ->limit(10)
            ->get();
        
        return view('coordenador.alunos.index', compact(
            'equipes',
            'statsAlunos',
            'alunos',
            'especialistas',
            'modulos',
            'distribuicaoStatus',
            'distribuicaoModulo',
            'status',
            'especialistaId',
            'modulo',
            'search'
        ));
    }
}

