<?php

namespace App\Http\Controllers\Coordenador;

use App\Http\Controllers\Controller;
use App\Models\Equipe;
use App\Models\Lead;
use App\Models\Usuario;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    /**
     * Exibe o dashboard do coordenador
     */
    public function index(Request $request)
    {
        $usuario = Auth::user();

        // Parâmetros de período
        $start = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $end = $request->input('end_date', Carbon::now()->format('Y-m-d'));

        // Buscar equipes do coordenador
        $equipes = Equipe::where('coordenador_id', $usuario->id)
            ->where('status', 'ativo')
            ->get();

        if ($equipes->isEmpty()) {
            return view('coordenador.dashboard', compact('start', 'end'))
                ->with('error', 'Você não está coordenando nenhuma equipe.');
        }

        $equipeIds = $equipes->pluck('id');

        // ============================================
        // KPIs DAS EQUIPES
        // ============================================

        // Total de especialistas
        $totalEspecialistas = DB::table('equipe_especialistas')
            ->whereIn('equipe_id', $equipeIds)
            ->where('status', 'ativo')
            ->distinct('especialista_id')
            ->count('especialista_id');

        // Leads ativos das equipes
        $leadsAtivos = Lead::whereIn('equipe_id', $equipeIds)
            ->whereIn('etapa', ['novo', 'contato', 'demo'])
            ->count();

        // Leads do período
        $leadsPeriodo = Lead::whereIn('equipe_id', $equipeIds)
            ->whereBetween(DB::raw('DATE(data_distribuicao)'), [$start, $end])
            ->count();

        // Buscar IDs dos especialistas das equipes
        $especialistaIds = DB::table('equipe_especialistas')
            ->whereIn('equipe_id', $equipeIds)
            ->where('status', 'ativo')
            ->pluck('especialista_id')
            ->toArray();

        // Vendas do período (alunos cadastrados)
        $vendasPeriodo = 0;
        $receitaPeriodo = 0;
        
        if (!empty($especialistaIds)) {
            $vendasPeriodo = DB::table('alunos')
                ->whereIn('especialista_id', $especialistaIds)
                ->whereBetween(DB::raw('DATE(data_cadastro)'), [$start, $end])
                ->where('status', 'ativo')
                ->count();

            $receitaPeriodo = DB::table('alunos')
                ->whereIn('especialista_id', $especialistaIds)
                ->whereBetween(DB::raw('DATE(data_cadastro)'), [$start, $end])
                ->where('status', 'ativo')
                ->sum(DB::raw('0'));
        }

        // Taxa de conversão
        $taxaConversao = $leadsPeriodo > 0 ? ($vendasPeriodo / $leadsPeriodo) * 100 : 0;

        // ============================================
        // PERFORMANCE POR EQUIPE
        // ============================================
        $performanceEquipes = [];
        foreach ($equipes as $equipe) {
            $especialistasEquipe = DB::table('equipe_especialistas')
                ->where('equipe_id', $equipe->id)
                ->where('status', 'ativo')
                ->count();

            $leadsEquipe = Lead::where('equipe_id', $equipe->id)
                ->whereBetween(DB::raw('DATE(data_distribuicao)'), [$start, $end])
                ->count();

            // Buscar IDs dos especialistas da equipe
            $espIdsEquipe = DB::table('equipe_especialistas')
                ->where('equipe_id', $equipe->id)
                ->where('status', 'ativo')
                ->pluck('especialista_id')
                ->toArray();

            $vendasEquipe = 0;
            $receitaEquipe = 0;
            
            if (!empty($espIdsEquipe)) {
                $vendasEquipe = DB::table('alunos')
                    ->whereIn('especialista_id', $espIdsEquipe)
                    ->whereBetween(DB::raw('DATE(data_cadastro)'), [$start, $end])
                    ->where('status', 'ativo')
                    ->count();

                $receitaEquipe = DB::table('alunos')
                    ->whereIn('especialista_id', $espIdsEquipe)
                    ->whereBetween(DB::raw('DATE(data_cadastro)'), [$start, $end])
                    ->where('status', 'ativo')
                    ->sum(DB::raw('0'));
            }

            $performanceEquipes[] = [
                'equipe' => $equipe,
                'especialistas' => $especialistasEquipe,
                'leads' => $leadsEquipe,
                'vendas' => $vendasEquipe,
                'receita' => $receitaEquipe,
                'taxa_conversao' => $leadsEquipe > 0 ? ($vendasEquipe / $leadsEquipe) * 100 : 0,
            ];
        }

        // ============================================
        // RANKING DE ESPECIALISTAS
        // ============================================
        $rankingEspecialistas = [];
        
        if (!empty($especialistaIds)) {
            $rankingEspecialistas = Usuario::whereIn('id', $especialistaIds)
                ->where('role', 'especialista')
                ->where('status', 'ativo')
                ->get()
                ->map(function ($especialista) use ($start, $end) {
                    $vendasPeriodo = DB::table('alunos')
                        ->where('especialista_id', $especialista->id)
                        ->whereBetween(DB::raw('DATE(data_cadastro)'), [$start, $end])
                        ->where('status', 'ativo')
                        ->count();

                    $receitaPeriodo = DB::table('alunos')
                        ->where('especialista_id', $especialista->id)
                        ->whereBetween(DB::raw('DATE(data_cadastro)'), [$start, $end])
                        ->where('status', 'ativo')
                        ->sum(DB::raw('0'));

                    return (object) [
                        'id' => $especialista->id,
                        'nome' => $especialista->nome,
                        'vendas_periodo' => $vendasPeriodo,
                        'receita_periodo' => $receitaPeriodo,
                    ];
                })
                ->sortByDesc('vendas_periodo')
                ->take(10)
                ->values();
        }

        // ============================================
        // EVOLUÇÃO DIÁRIA
        // ============================================
        $evolucaoDiaria = Lead::selectRaw('DATE(data_distribuicao) as data, COUNT(*) as total')
            ->whereIn('equipe_id', $equipeIds)
            ->whereBetween(DB::raw('DATE(data_distribuicao)'), [$start, $end])
            ->groupBy('data')
            ->orderBy('data')
            ->get();

        return view('coordenador.dashboard', compact(
            'equipes',
            'start',
            'end',
            'totalEspecialistas',
            'leadsAtivos',
            'leadsPeriodo',
            'vendasPeriodo',
            'receitaPeriodo',
            'taxaConversao',
            'performanceEquipes',
            'rankingEspecialistas',
            'evolucaoDiaria'
        ));
    }
}

