<?php

namespace App\Http\Controllers\Coordenador;

use App\Http\Controllers\Controller;
use App\Models\Equipe;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class EquipeController extends Controller
{
    /**
     * Exibe a página de gerenciamento da equipe
     */
    public function index()
    {
        $coordenador = Auth::user();
        
        // Buscar equipes do coordenador
        $equipes = Equipe::where('coordenador_id', $coordenador->id)
            ->where('status', 'ativo')
            ->get();
        
        if ($equipes->isEmpty()) {
            // Criar equipe padrão se não existir
            $equipe = Equipe::create([
                'nome' => 'Equipe ' . $coordenador->nome,
                'coordenador_id' => $coordenador->id,
                'meta_mensal' => 0,
                'status' => 'ativo',
            ]);
            
            $equipes = collect([$equipe]);
        }
        
        $equipe = $equipes->first();
        $equipeId = $equipe->id;
        
        // Estatísticas da equipe
        $statsEquipe = DB::table('equipe_especialistas as ee')
            ->join('equipes as e', 'ee.equipe_id', '=', 'e.id')
            ->leftJoin('usuarios as u', 'ee.especialista_id', '=', 'u.id')
            ->leftJoin('alunos as a', 'u.id', '=', 'a.especialista_id')
            ->where('e.id', $equipeId)
            ->where('ee.status', 'ativo')
            ->selectRaw('
                COUNT(DISTINCT ee.especialista_id) as total_membros,
                COUNT(DISTINCT CASE WHEN u.role = "especialista" THEN ee.especialista_id END) as total_especialistas,
                COUNT(DISTINCT a.id) as total_alunos,
                COUNT(DISTINCT CASE WHEN a.status = "ativo" THEN a.id END) as alunos_ativos,
                COUNT(DISTINCT CASE WHEN a.data_conclusao IS NOT NULL THEN a.id END) as alunos_concluidos,
                COUNT(DISTINCT CASE WHEN MONTH(a.data_cadastro) = MONTH(CURRENT_DATE()) 
                                   AND YEAR(a.data_cadastro) = YEAR(CURRENT_DATE()) THEN a.id END) as vendas_mes
            ')
            ->first();
        
        // Membros da equipe
        $membros = DB::table('equipe_especialistas as ee')
            ->join('usuarios as u', 'ee.especialista_id', '=', 'u.id')
            ->leftJoin('alunos as a', 'u.id', '=', 'a.especialista_id')
            ->where('ee.equipe_id', $equipeId)
            ->where('ee.status', 'ativo')
            ->select(
                'ee.id as usuario_equipe_id',
                'u.id',
                'u.nome',
                'u.email',
                'u.role as categoria',
                'ee.data_entrada',
                DB::raw('COUNT(a.id) as total_alunos'),
                DB::raw('COUNT(CASE WHEN a.status = "ativo" THEN 1 END) as alunos_ativos'),
                DB::raw('COUNT(CASE WHEN MONTH(a.data_cadastro) = MONTH(CURRENT_DATE()) 
                                   AND YEAR(a.data_cadastro) = YEAR(CURRENT_DATE()) THEN 1 END) as vendas_mes')
            )
            ->groupBy('ee.id', 'u.id', 'u.nome', 'u.email', 'u.role', 'ee.data_entrada')
            ->orderBy('u.role')
            ->orderBy('u.nome')
            ->get();
        
        // Histórico de vendas dos últimos 6 meses
        $historicoVendas = DB::table('alunos as a')
            ->join('equipe_especialistas as ee', 'a.especialista_id', '=', 'ee.especialista_id')
            ->where('ee.equipe_id', $equipeId)
            ->where('ee.status', 'ativo')
            ->where('a.data_cadastro', '>=', DB::raw('DATE_SUB(CURRENT_DATE(), INTERVAL 6 MONTH)'))
            ->selectRaw('
                DATE_FORMAT(a.data_cadastro, "%Y-%m") as mes,
                COUNT(*) as vendas
            ')
            ->groupBy('mes')
            ->orderBy('mes')
            ->get();
        
        // Calcular progresso da meta mensal
        $progressoMeta = 0;
        if ($equipe->meta_mensal > 0) {
            $progressoMeta = ($statsEquipe->vendas_mes / $equipe->meta_mensal) * 100;
        }
        
        return view('coordenador.equipes.index', compact(
            'equipe',
            'statsEquipe',
            'membros',
            'historicoVendas',
            'progressoMeta'
        ));
    }
    
    /**
     * Atualizar informações da equipe
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'nome' => 'required|string|max:255',
            'meta_mensal' => 'nullable|integer|min:0',
            'descricao' => 'nullable|string',
        ]);
        
        $coordenador = Auth::user();
        
        // Verificar se a equipe pertence ao coordenador
        $equipe = Equipe::where('id', $id)
            ->where('coordenador_id', $coordenador->id)
            ->firstOrFail();
        
        // Atualizar equipe
        $equipe->update([
            'nome' => $request->nome,
            'meta_mensal' => $request->meta_mensal ?? 0,
            'descricao' => $request->descricao,
        ]);
        
        return back()->with('success', 'Equipe atualizada com sucesso!');
    }
}

