<?php

namespace App\Http\Controllers\Coordenador;

use App\Http\Controllers\Controller;
use App\Models\Equipe;
use App\Models\Usuario;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class EspecialistaController extends Controller
{
    /**
     * Exibe a lista de especialistas da equipe
     */
    public function index()
    {
        $coordenador = Auth::user();
        
        // Buscar equipes do coordenador
        $equipes = Equipe::where('coordenador_id', $coordenador->id)
            ->where('status', 'ativo')
            ->get();
        
        if ($equipes->isEmpty()) {
            return redirect()->route('coordenador.dashboard')
                ->with('error', 'Você não possui equipes associadas.');
        }
        
        $equipeIds = $equipes->pluck('id');
        
        // Especialistas da equipe
        $especialistasEquipe = DB::table('equipe_especialistas as ee')
            ->join('usuarios as u', 'ee.especialista_id', '=', 'u.id')
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo')
            ->where('u.role', 'especialista')
            ->where('u.status', 'ativo')
            ->select(
                'ee.id as usuario_equipe_id',
                'u.id',
                'u.nome',
                'u.email',
                'ee.data_entrada',
                DB::raw('(SELECT COUNT(*) FROM alunos WHERE especialista_id = u.id) as total_alunos'),
                DB::raw('(SELECT COUNT(*) FROM alunos WHERE especialista_id = u.id AND status = "ativo") as alunos_ativos'),
                DB::raw('(SELECT COUNT(*) FROM alunos WHERE especialista_id = u.id AND data_conclusao IS NOT NULL) as alunos_concluidos'),
                DB::raw('(SELECT COUNT(*) FROM alunos WHERE especialista_id = u.id AND status = "parado") as alunos_parados'),
                DB::raw('ROUND((SELECT COUNT(*) FROM alunos WHERE especialista_id = u.id AND data_conclusao IS NOT NULL) / NULLIF((SELECT COUNT(*) FROM alunos WHERE especialista_id = u.id), 0) * 100, 1) as taxa_conclusao')
            )
            ->orderByRaw('(SELECT COUNT(*) FROM alunos WHERE especialista_id = u.id) DESC')
            ->get();
        
        // Especialistas disponíveis (não estão em nenhuma equipe)
        $especialistasDisponiveis = Usuario::where('role', 'especialista')
            ->where('status', 'ativo')
            ->whereNotIn('id', function($query) {
                $query->select('especialista_id')
                    ->from('equipe_especialistas')
                    ->where('status', 'ativo');
            })
            ->orderBy('nome')
            ->get();
        
        return view('coordenador.especialistas.index', compact(
            'equipes',
            'especialistasEquipe',
            'especialistasDisponiveis'
        ));
    }
    
    /**
     * Adicionar especialista à equipe
     */
    public function adicionar(Request $request)
    {
        $request->validate([
            'especialista_id' => 'required|exists:usuarios,id',
            'equipe_id' => 'required|exists:equipes,id',
        ]);
        
        $coordenador = Auth::user();
        
        // Verificar se a equipe pertence ao coordenador
        $equipe = Equipe::where('id', $request->equipe_id)
            ->where('coordenador_id', $coordenador->id)
            ->first();
        
        if (!$equipe) {
            return back()->withErrors(['error' => 'Equipe não encontrada ou não pertence a você.']);
        }
        
        // Verificar se já está na equipe
        $existe = DB::table('equipe_especialistas')
            ->where('equipe_id', $request->equipe_id)
            ->where('especialista_id', $request->especialista_id)
            ->where('status', 'ativo')
            ->exists();
        
        if ($existe) {
            return back()->with('warning', 'Especialista já está nesta equipe.');
        }
        
        // Adicionar à equipe
        DB::table('equipe_especialistas')->insert([
            'equipe_id' => $request->equipe_id,
            'especialista_id' => $request->especialista_id,
            'data_entrada' => now(),
            'status' => 'ativo',
            'criado_em' => now(),
            'atualizado_em' => now(),
        ]);
        
        return back()->with('success', 'Especialista adicionado à equipe com sucesso!');
    }
    
    /**
     * Remover especialista da equipe
     */
    public function remover(Request $request)
    {
        $request->validate([
            'usuario_equipe_id' => 'required|exists:equipe_especialistas,id',
        ]);
        
        $coordenador = Auth::user();
        
        // Verificar se o vínculo pertence a uma equipe do coordenador
        $vinculo = DB::table('equipe_especialistas as ee')
            ->join('equipes as e', 'ee.equipe_id', '=', 'e.id')
            ->where('ee.id', $request->usuario_equipe_id)
            ->where('e.coordenador_id', $coordenador->id)
            ->first();
        
        if (!$vinculo) {
            return back()->withErrors(['error' => 'Vínculo não encontrado ou não pertence às suas equipes.']);
        }
        
        // Remover da equipe (desativar)
        DB::table('equipe_especialistas')
            ->where('id', $request->usuario_equipe_id)
            ->update([
                'status' => 'inativo',
                'data_saida' => now(),
                'atualizado_em' => now(),
            ]);
        
        return back()->with('success', 'Especialista removido da equipe com sucesso!');
    }
}

