<?php

namespace App\Http\Controllers\Coordenador;

use App\Http\Controllers\Controller;
use App\Models\Equipe;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class VendaController extends Controller
{
    /**
     * Exibe a página de acompanhamento de vendas
     */
    public function index(Request $request)
    {
        $coordenador = Auth::user();
        
        // Buscar equipes do coordenador
        $equipes = Equipe::where('coordenador_id', $coordenador->id)
            ->where('status', 'ativo')
            ->get();
        
        if ($equipes->isEmpty()) {
            return redirect()->route('coordenador.dashboard')
                ->with('error', 'Você não possui equipes associadas.');
        }
        
        $equipeIds = $equipes->pluck('id');
        
        // Filtros
        $mes = $request->get('mes', date('m'));
        $ano = $request->get('ano', date('Y'));
        $especialistaId = $request->get('especialista_id', '');
        
        // Estatísticas gerais de vendas
        $statsGerais = DB::table('alunos as a')
            ->join('equipe_especialistas as ee', 'a.especialista_id', '=', 'ee.especialista_id')
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo')
            ->selectRaw('
                COUNT(*) as total_vendas,
                SUM(CASE WHEN MONTH(a.data_cadastro) = ' . (int)$mes . ' AND YEAR(a.data_cadastro) = ' . (int)$ano . ' THEN 1 ELSE 0 END) as vendas_mes,
                SUM(CASE WHEN MONTH(a.data_cadastro) = MONTH(DATE_SUB(CURRENT_DATE(), INTERVAL 1 MONTH)) 
                       AND YEAR(a.data_cadastro) = YEAR(DATE_SUB(CURRENT_DATE(), INTERVAL 1 MONTH)) THEN 1 ELSE 0 END) as vendas_mes_anterior,
                SUM(CASE WHEN a.status = "ativo" THEN 1 ELSE 0 END) as alunos_ativos,
                SUM(CASE WHEN a.data_conclusao IS NOT NULL THEN 1 ELSE 0 END) as alunos_concluidos,
                SUM(CASE WHEN a.status = "trancado" THEN 1 ELSE 0 END) as alunos_parados
            ')
            ->first();
        
        // Calcular variação percentual
        $variacao = 0;
        if ($statsGerais->vendas_mes_anterior > 0) {
            $variacao = (($statsGerais->vendas_mes - $statsGerais->vendas_mes_anterior) / $statsGerais->vendas_mes_anterior) * 100;
        }
        
        // Query base para vendas por especialista
        $query = DB::table('usuarios as u')
            ->join('equipe_especialistas as ee', 'u.id', '=', 'ee.especialista_id')
            ->join('equipes as e', 'ee.equipe_id', '=', 'e.id')
            ->leftJoin('alunos as a', 'u.id', '=', 'a.especialista_id')
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo')
            ->where('u.role', 'especialista')
            ->where('u.status', 'ativo');
        
        if ($especialistaId) {
            $query->where('a.especialista_id', $especialistaId);
        }
        
        // Vendas por especialista
        $vendasEspecialistas = $query->select(
                'u.id',
                'u.nome',
                DB::raw('COUNT(a.id) as total_vendas'),
                DB::raw('SUM(CASE WHEN MONTH(a.data_cadastro) = ' . (int)$mes . ' AND YEAR(a.data_cadastro) = ' . (int)$ano . ' THEN 1 ELSE 0 END) as vendas_mes'),
                DB::raw('SUM(CASE WHEN a.status = "ativo" THEN 1 ELSE 0 END) as alunos_ativos'),
                DB::raw('SUM(CASE WHEN a.data_conclusao IS NOT NULL THEN 1 ELSE 0 END) as alunos_concluidos'),
                DB::raw('ROUND((SUM(CASE WHEN a.data_conclusao IS NOT NULL THEN 1 ELSE 0 END) / NULLIF(COUNT(a.id), 0)) * 100, 1) as taxa_conclusao'),
                DB::raw('0 as valor_medio_curso')
            )
            ->groupBy('u.id', 'u.nome')
            ->orderByDesc('vendas_mes')
            ->orderByDesc('total_vendas')
            ->get();
        
        // Vendas por dia do mês selecionado
        $vendasDiarias = DB::table('alunos as a')
            ->join('equipe_especialistas as ee', 'a.especialista_id', '=', 'ee.especialista_id')
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo')
            ->whereRaw('MONTH(a.data_cadastro) = ? AND YEAR(a.data_cadastro) = ?', [$mes, $ano])
            ->selectRaw('DATE(a.data_cadastro) as data, COUNT(*) as vendas')
            ->groupBy('data')
            ->orderBy('data')
            ->get();
        
        // Vendas dos últimos 12 meses
        $vendas12Meses = DB::table('alunos as a')
            ->join('equipe_especialistas as ee', 'a.especialista_id', '=', 'ee.especialista_id')
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo')
            ->where('a.data_cadastro', '>=', DB::raw('DATE_SUB(CURRENT_DATE(), INTERVAL 12 MONTH)'))
            ->selectRaw('DATE_FORMAT(a.data_cadastro, "%Y-%m") as mes, COUNT(*) as vendas')
            ->groupBy('mes')
            ->orderBy('mes')
            ->get();
        
        // Especialistas para filtro
        $especialistas = DB::table('equipe_especialistas as ee')
            ->join('usuarios as u', 'ee.especialista_id', '=', 'u.id')
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo')
            ->where('u.role', 'especialista')
            ->where('u.status', 'ativo')
            ->select('u.id', 'u.nome')
            ->orderBy('u.nome')
            ->get();
        
        return view('coordenador.vendas.index', compact(
            'equipes',
            'statsGerais',
            'variacao',
            'vendasEspecialistas',
            'vendasDiarias',
            'vendas12Meses',
            'especialistas',
            'mes',
            'ano',
            'especialistaId'
        ));
    }

    /**
     * Lista negociações aguardando aprovação da equipe
     */
    public function aguardandoAprovacao(Request $request)
    {
        $coordenador = Auth::user();
        
        // Buscar equipes do coordenador
        $equipes = Equipe::where('coordenador_id', $coordenador->id)
            ->where('status', 'ativo')
            ->get();
        
        if ($equipes->isEmpty()) {
            return redirect()->route('coordenador.dashboard')
                ->with('error', 'Você não possui equipes associadas.');
        }
        
        $equipeIds = $equipes->pluck('id');
        
        // Buscar especialistas das equipes
        $especialistaIds = DB::table('equipe_especialistas')
            ->whereIn('equipe_id', $equipeIds)
            ->where('status', 'ativo')
            ->pluck('especialista_id');
        
        // Buscar negociações aguardando aprovação (status 'andamento' e lead com etapa 'fechamento')
        $query = DB::table('negociacoes as n')
            ->join('leads as l', 'n.lead_id', '=', 'l.id')
            ->leftJoin('usuarios as e', 'n.especialista_id', '=', 'e.id')
            ->where('n.status', 'andamento')
            ->where('l.etapa', 'fechamento')
            ->whereIn('n.especialista_id', $especialistaIds)
            ->select(
                'n.id',
                'n.lead_id',
                'n.especialista_id',
                'n.valor_proposta',
                'n.valor_total',
                'n.data_criacao',
                'n.observacoes',
                'l.nome as lead_nome',
                'l.email as lead_email',
                'l.telefone as lead_telefone',
                'e.nome as especialista_nome'
            );

        // Filtros
        if ($request->filled('especialista_id')) {
            $query->where('n.especialista_id', $request->especialista_id);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('l.nome', 'like', "%{$search}%")
                  ->orWhere('l.email', 'like', "%{$search}%")
                  ->orWhere('l.telefone', 'like', "%{$search}%");
            });
        }

        $vendas = $query->orderBy('n.data_criacao', 'desc')->paginate(50);

        // Estatísticas
        $stats = [
            'aguardando_aprovacao' => DB::table('negociacoes as n')
                ->join('leads as l', 'n.lead_id', '=', 'l.id')
                ->where('n.status', 'andamento')
                ->where('l.etapa', 'fechamento')
                ->whereIn('n.especialista_id', $especialistaIds)
                ->count(),
        ];

        // Especialistas para filtro
        $especialistas = DB::table('equipe_especialistas as ee')
            ->join('usuarios as u', 'ee.especialista_id', '=', 'u.id')
            ->whereIn('ee.equipe_id', $equipeIds)
            ->where('ee.status', 'ativo')
            ->where('u.role', 'especialista')
            ->where('u.status', 'ativo')
            ->select('u.id', 'u.nome')
            ->orderBy('u.nome')
            ->get();

        return view('coordenador.vendas.aguardando-aprovacao', compact('vendas', 'stats', 'especialistas'));
    }

    /**
     * Aprova uma venda e cria a matrícula (coordenador)
     */
    public function aprovar(Request $request, $id)
    {
        $request->validate([
            'curso' => 'required|string|max:150',
            'valor_total' => 'required|numeric|min:0',
        ], [
            'curso.required' => 'Informe o curso.',
            'valor_total.required' => 'Informe o valor total.',
        ]);

        $coordenador = Auth::user();
        $now = Carbon::now('America/Sao_Paulo');
        
        DB::beginTransaction();
        try {
            // Buscar negociação
            $negociacao = DB::table('negociacoes')
                ->where('id', $id)
                ->where('status', 'andamento')
                ->first();

            if (!$negociacao) {
                return response()->json([
                    'success' => false,
                    'message' => 'Esta negociação não está aguardando aprovação.',
                ], 400);
            }

            // Verificar se o especialista pertence a uma equipe do coordenador
            $equipes = Equipe::where('coordenador_id', $coordenador->id)
                ->where('status', 'ativo')
                ->pluck('id');
            
            $pertenceEquipe = DB::table('equipe_especialistas')
                ->whereIn('equipe_id', $equipes)
                ->where('especialista_id', $negociacao->especialista_id)
                ->where('status', 'ativo')
                ->exists();

            if (!$pertenceEquipe) {
                return response()->json([
                    'success' => false,
                    'message' => 'Você não tem permissão para aprovar esta negociação.',
                ], 403);
            }

            // Buscar lead
            $lead = DB::table('leads')->where('id', $negociacao->lead_id)->first();
            
            if (!$lead) {
                return response()->json([
                    'success' => false,
                    'message' => 'Lead não encontrado.',
                ], 404);
            }

            // Verificar se já existe aluno para este lead
            $aluno = DB::table('alunos')
                ->where('email', $lead->email)
                ->orWhere('cpf', $lead->cpf)
                ->first();

            // Se não existir aluno, criar
            if (!$aluno) {
                $alunoId = DB::table('alunos')->insertGetId([
                    'nome' => $lead->nome,
                    'email' => $lead->email,
                    'telefone' => $lead->telefone,
                    'cpf' => $lead->cpf ?? null,
                    'status' => 'ativo',
                    'especialista_id' => $negociacao->especialista_id,
                    'data_cadastro' => $now,
                    'tipo_contrato' => 'regular',
                ]);
            } else {
                $alunoId = $aluno->id;
            }

            // Criar vínculo entre lead e aluno
            $vinculoExistente = DB::table('vinculo')
                ->where('lead_id', $lead->id)
                ->where('aluno_id', $alunoId)
                ->first();

            if (!$vinculoExistente) {
                DB::table('vinculo')->insert([
                    'lead_id' => $lead->id,
                    'aluno_id' => $alunoId,
                    'especialista_id' => $negociacao->especialista_id,
                    'status' => 'ativo',
                    'data_criacao' => $now,
                ]);
            }

            // Criar matrícula
            $matriculaId = DB::table('matriculas')->insertGetId([
                'aluno_id' => $alunoId,
                'especialista_id' => $negociacao->especialista_id,
                'curso' => $request->curso,
                'valor_total' => $request->valor_total,
                'status' => 'ativa',
                'data_matricula' => $now->format('Y-m-d'),
            ]);

            // Atualizar negociação
            DB::table('negociacoes')
                ->where('id', $id)
                ->update([
                    'status' => 'fechado',
                    'aluno_id' => $alunoId,
                    'valor_total' => $request->valor_total,
                    'data_fechamento' => $now,
                    'aprovado_por' => $coordenador->id,
                ]);

            // Atualizar lead
            DB::table('leads')
                ->where('id', $lead->id)
                ->update([
                    'status' => 'convertido',
                    'etapa' => 'convertido',
                ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Negociação aprovada e matrícula criada com sucesso!',
                'matricula_id' => $matriculaId,
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Erro ao aprovar: ' . $e->getMessage(),
            ], 500);
        }
    }
}

