<?php

namespace App\Http\Controllers\Especialista;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class NegociacaoModalController extends Controller
{
    /**
     * Retorna o HTML do modal de negociação para um lead
     */
    public function getModal(Request $request, $leadId)
    {
        $especialista = Auth::user();
        
        // Buscar dados do lead
        $lead = DB::table('leads')
            ->where('id', $leadId)
            ->where(function($query) use ($especialista) {
                $query->where('especialista_id', $especialista->id)
                      ->orWhereNull('especialista_id');
            })
            ->first();
        
        if (!$lead) {
            return response()->json([
                'error' => 'Lead não encontrado ou não disponível para você.'
            ], 404);
        }
        
        // Buscar valor integral ativo
        $valorCurso = DB::table('valores_curso')
            ->where('ativo', 1)
            ->orderBy('data_criacao', 'desc')
            ->first();
        
        $valorIntegral = $valorCurso ? (float)$valorCurso->valor_integral : 5000.00;
        
        // Buscar alunos disponíveis (não vinculados) do especialista
        $alunos = DB::table('alunos as a')
            ->leftJoin('vinculo as v', function($join) {
                $join->on('a.id', '=', 'v.aluno_id')
                     ->where('v.status', '=', 'ativo');
            })
            ->where('a.especialista_id', $especialista->id)
            ->where('a.status', 'ativo')
            ->whereNull('v.id')
            ->select('a.id', 'a.nome', 'a.email', 'a.celular', 'a.data_cadastro', 'a.tipo_contrato')
            ->orderBy('a.data_cadastro', 'desc')
            ->limit(100)
            ->get();
        
        // Buscar vouchers vigentes do especialista
        $hoje = Carbon::now('America/Sao_Paulo')->format('Y-m-d');
        
        // Buscar vouchers globais (disponíveis para todos)
        $vouchersGlobais = DB::table('vouchers as v')
            ->leftJoin('voucher_especialista as ve', function($join) use ($especialista) {
                $join->on('v.id', '=', 've.voucher_id')
                     ->where('ve.especialista_id', '=', $especialista->id);
            })
            ->where('v.ativo', 1)
            ->where('v.status', 'ativo')
            ->where(function($query) use ($hoje) {
                $query->whereNull('v.validade')
                      ->orWhere('v.validade', '>=', $hoje);
            })
            ->whereRaw('v.quantidade_usada < v.quantidade_total')
            ->where(function($query) use ($especialista) {
                // Vouchers globais (sem vínculo específico) ou vinculados a este especialista
                $query->whereNull('ve.id')
                      ->orWhere('ve.especialista_id', $especialista->id);
            })
            ->select(
                'v.id',
                'v.codigo',
                'v.percentual_desconto',
                'v.tipo',
                'v.validade as valido_ate',
                'v.quantidade_total',
                'v.quantidade_usada'
            )
            ->get();
        
        // Buscar vouchers individuais (vinculados a este especialista)
        $vouchersIndividuais = DB::table('vouchers as v')
            ->join('voucher_especialista as ve', function($join) use ($especialista) {
                $join->on('v.id', '=', 've.voucher_id')
                     ->where('ve.especialista_id', '=', $especialista->id)
                     ->where('ve.status', '=', 'ativo');
            })
            ->where('v.ativo', 1)
            ->where('v.status', 'ativo')
            ->where(function($query) use ($hoje) {
                $query->whereNull('v.validade')
                      ->orWhere('v.validade', '>=', $hoje);
            })
            ->whereRaw('v.quantidade_usada < v.quantidade_total')
            ->select(
                'v.id',
                'v.codigo',
                'v.percentual_desconto',
                'v.tipo',
                'v.validade as valido_ate',
                'v.quantidade_total',
                'v.quantidade_usada'
            )
            ->get();
        
        // Combinar e remover duplicatas
        $vouchers = $vouchersGlobais->merge($vouchersIndividuais)->unique('id')->values();
        
        // Verificar se já existe vínculo para este lead
        $vinculoExistente = DB::table('vinculo')
            ->where('lead_id', $leadId)
            ->where('status', 'ativo')
            ->first();
        
        $html = view('especialista.atendimentos.modal-negociacao', compact(
            'lead',
            'alunos',
            'vouchers',
            'valorIntegral',
            'vinculoExistente'
        ))->render();
        
        return response()->json([
            'html' => $html
        ]);
    }
    
    /**
     * Salva a negociação e gera o link de pagamento
     */
    public function salvarNegociacao(Request $request)
    {
        $especialista = Auth::user();
        $now = Carbon::now('America/Sao_Paulo');
        
        // Normalizar dados antes da validação
        $data = $request->all();
        
        // Garantir que tipo_valor e forma_pagamento sejam strings válidas PRIMEIRO
        if (!isset($data['tipo_valor']) || $data['tipo_valor'] === '' || $data['tipo_valor'] === null || !is_string($data['tipo_valor'])) {
            $data['tipo_valor'] = 'integral';
        }
        $data['tipo_valor'] = trim((string) $data['tipo_valor']);
        
        if (!isset($data['forma_pagamento']) || $data['forma_pagamento'] === '' || $data['forma_pagamento'] === null || !is_string($data['forma_pagamento'])) {
            return response()->json([
                'success' => false,
                'message' => 'A forma de pagamento é obrigatória.',
                'errors' => ['forma_pagamento' => ['A forma de pagamento é obrigatória.']],
                'debug' => [
                    'forma_pagamento_received' => $data['forma_pagamento'] ?? 'NÃO DEFINIDO',
                    'forma_pagamento_type' => isset($data['forma_pagamento']) ? gettype($data['forma_pagamento']) : 'N/A',
                ]
            ], 422);
        }
        $data['forma_pagamento'] = trim((string) $data['forma_pagamento']);
        
        // Converter tipos numéricos
        if (isset($data['lead_id'])) {
            $data['lead_id'] = (int) $data['lead_id'];
        }
        if (isset($data['aluno_existente_id'])) {
            $data['aluno_existente_id'] = (int) $data['aluno_existente_id'];
        }
        if (isset($data['num_parcelas'])) {
            $data['num_parcelas'] = (int) $data['num_parcelas'];
        }
        if (isset($data['voucher_id']) && $data['voucher_id'] !== null && $data['voucher_id'] !== '') {
            $data['voucher_id'] = (int) $data['voucher_id'];
        } else {
            $data['voucher_id'] = null;
        }
        if (isset($data['investimento_parcial'])) {
            $data['investimento_parcial'] = filter_var($data['investimento_parcial'], FILTER_VALIDATE_BOOLEAN);
        } else {
            $data['investimento_parcial'] = false;
        }
        if (isset($data['valor_investido']) && $data['valor_investido'] !== null && $data['valor_investido'] !== '') {
            $data['valor_investido'] = (float) $data['valor_investido'];
        } else {
            $data['valor_investido'] = null;
        }
        if (isset($data['data_segunda_parcela']) && $data['data_segunda_parcela'] === '') {
            $data['data_segunda_parcela'] = null;
        }
        
        // Criar novo request com dados normalizados
        $request->merge($data);
        
        // Criar novo request com dados normalizados
        $request->merge($data);
        
        try {
            // Validar campos um por um para identificar qual está falhando
            $validator = \Validator::make($data, [
                'lead_id' => 'required|integer|exists:leads,id',
                'aluno_existente_id' => 'required|integer|exists:alunos,id',
                'tipo_valor' => 'required|string|in:integral,bolsista',
                'forma_pagamento' => 'required|string|in:credito,pix',
                'num_parcelas' => 'required|integer|min:1|max:12',
                'voucher_id' => 'nullable|integer|exists:vouchers,id',
                'investimento_parcial' => 'nullable|boolean',
                'valor_investido' => 'nullable|numeric|min:0',
                'data_segunda_parcela' => 'nullable|date',
            ], [
                'lead_id.required' => 'O ID do lead é obrigatório.',
                'lead_id.exists' => 'Lead não encontrado.',
                'aluno_existente_id.required' => 'Por favor, selecione um aluno.',
                'aluno_existente_id.exists' => 'Aluno não encontrado.',
                'tipo_valor.required' => 'O tipo de valor é obrigatório.',
                'tipo_valor.string' => 'O tipo de valor deve ser uma string.',
                'tipo_valor.in' => 'Tipo de valor inválido.',
                'forma_pagamento.required' => 'A forma de pagamento é obrigatória.',
                'forma_pagamento.string' => 'A forma de pagamento deve ser uma string.',
                'forma_pagamento.in' => 'Forma de pagamento inválida.',
                'num_parcelas.required' => 'O número de parcelas é obrigatório.',
                'num_parcelas.integer' => 'O número de parcelas deve ser um número inteiro.',
                'num_parcelas.min' => 'O número de parcelas deve ser no mínimo 1.',
                'num_parcelas.max' => 'O número de parcelas deve ser no máximo 12.',
                'voucher_id.exists' => 'Voucher não encontrado.',
                'valor_investido.numeric' => 'O valor investido deve ser um número.',
                'data_segunda_parcela.date' => 'A data da segunda parcela deve ser uma data válida.',
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Erro de validação: ' . implode(', ', $validator->errors()->all()),
                    'errors' => $validator->errors(),
                    'received_data' => $data,
                    'debug' => [
                        'tipo_valor' => $data['tipo_valor'] ?? 'NÃO DEFINIDO',
                        'tipo_valor_type' => isset($data['tipo_valor']) ? gettype($data['tipo_valor']) : 'N/A',
                        'forma_pagamento' => $data['forma_pagamento'] ?? 'NÃO DEFINIDO',
                        'forma_pagamento_type' => isset($data['forma_pagamento']) ? gettype($data['forma_pagamento']) : 'N/A',
                    ]
                ], 422);
            }
            
            $validated = $validator->validated();
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro inesperado: ' . $e->getMessage(),
                'errors' => ['geral' => [$e->getMessage()]],
                'received_data' => $data
            ], 500);
        }
        
        DB::beginTransaction();
        try {
            // Verificar se o lead pertence ao especialista
            $lead = DB::table('leads')
                ->where('id', $request->lead_id)
                ->where(function($query) use ($especialista) {
                    $query->where('especialista_id', $especialista->id)
                          ->orWhereNull('especialista_id');
                })
                ->first();
            
            if (!$lead) {
                throw new \Exception('Lead não encontrado ou não disponível para você.');
            }
            
            // Verificar se o aluno pertence ao especialista
            $aluno = DB::table('alunos')
                ->where('id', $request->aluno_existente_id)
                ->where('especialista_id', $especialista->id)
                ->first();
            
            if (!$aluno) {
                throw new \Exception('Aluno não encontrado ou não pertence a você.');
            }
            
            // Buscar valor do curso
            $valorCurso = DB::table('valores_curso')
                ->where('ativo', 1)
                ->orderBy('data_criacao', 'desc')
                ->first();
            
            $valorBase = $valorCurso ? (float)$valorCurso->valor_integral : 5000.00;
            
            // Aplicar desconto do voucher
            $valorTotal = $valorBase;
            $voucherId = $request->voucher_id;
            $percentualDesconto = 0;
            
            if ($voucherId) {
                $voucher = DB::table('vouchers')
                    ->where('id', $voucherId)
                    ->where('ativo', 1)
                    ->first();
                
                if ($voucher && $voucher->percentual_desconto) {
                    $percentualDesconto = (float)$voucher->percentual_desconto;
                    $valorTotal = $valorBase * (1 - $percentualDesconto / 100);
                }
            }
            
            // Aplicar juros se cartão parcelado
            if ($request->forma_pagamento === 'credito' && $request->num_parcelas > 1) {
                $juros = 0.0199; // 1.99% ao mês
                $valorTotal = $valorTotal * pow(1 + $juros, $request->num_parcelas);
            }
            
            // Se investimento parcial, usar valor investido
            if ($request->investimento_parcial && $request->valor_investido) {
                $valorTotal = (float)$request->valor_investido;
            }
            
            // Criar ou atualizar vínculo
            $vinculo = DB::table('vinculo')
                ->where('lead_id', $request->lead_id)
                ->where('status', 'ativo')
                ->first();
            
            if (!$vinculo) {
                $vinculoId = DB::table('vinculo')->insertGetId([
                    'lead_id' => $request->lead_id,
                    'aluno_id' => $request->aluno_existente_id,
                    'especialista_id' => $especialista->id,
                    'status' => 'ativo',
                    'data_criacao' => $now,
                ]);
            } else {
                $vinculoId = $vinculo->id;
                // Atualizar vínculo existente
                DB::table('vinculo')
                    ->where('id', $vinculoId)
                    ->update([
                        'aluno_id' => $request->aluno_existente_id,
                        'data_atualizacao' => $now,
                    ]);
            }
            
            // Criar negociação
            $negociacaoId = DB::table('negociacoes')->insertGetId([
                'lead_id' => $request->lead_id,
                'aluno_id' => $request->aluno_existente_id,
                'especialista_id' => $especialista->id,
                'valor_proposta' => $valorTotal,
                'valor_total' => $valorTotal,
                'num_parcelas' => $request->num_parcelas,
                'status' => 'andamento',
                'data_criacao' => $now,
                'criado_por' => $especialista->id,
                'voucher_id' => $voucherId,
            ]);
            
            // Atualizar voucher se usado
            if ($voucherId) {
                // Incrementar quantidade_usada na tabela vouchers
                DB::table('vouchers')
                    ->where('id', $voucherId)
                    ->increment('quantidade_usada');
            }
            
            // Atualizar status do lead para "demo" (negociação)
            DB::table('leads')
                ->where('id', $request->lead_id)
                ->update([
                    'etapa' => 'demo',
                    'status' => 'negociacao',
                    'data_atualizacao' => $now,
                    'especialista_id' => $especialista->id,
                ]);
            
            // Gerar link de pagamento (por enquanto, link temporário)
            // TODO: Integrar com Pagar.me quando necessário
            $paymentLink = url("/pagamento/{$negociacaoId}");
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Negociação criada com sucesso!',
                'data' => [
                    'negociacao_id' => $negociacaoId,
                    'lead_id' => $request->lead_id,
                    'aluno_id' => $request->aluno_existente_id,
                    'valor_total' => $valorTotal,
                    'payment_link' => $paymentLink,
                    'order_id' => 'N/A', // Será preenchido quando integrar Pagar.me
                ]
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Erro ao criar negociação: ' . $e->getMessage()
            ], 500);
        }
    }
}

