<?php

namespace App\Http\Controllers\Especialista;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class PagamentoController extends Controller
{
    /**
     * Confirma pagamento e atualiza status do lead automaticamente
     * Este método pode ser chamado via webhook do Pagar.me ou manualmente
     */
    public function confirmarPagamento(Request $request, $negociacaoId)
    {
        $now = Carbon::now('America/Sao_Paulo');
        
        // Buscar negociação
        $negociacao = DB::table('negociacoes')
            ->where('id', $negociacaoId)
            ->first();
        
        if (!$negociacao) {
            return response()->json([
                'success' => false,
                'message' => 'Negociação não encontrada.'
            ], 404);
        }
        
        // Validar dados do pagamento
        $request->validate([
            'valor_pago' => 'required|numeric|min:0',
            'data_pagamento' => 'nullable|date',
        ]);
        
        $valorPago = (float) $request->valor_pago;
        $dataPagamento = $request->data_pagamento ? Carbon::parse($request->data_pagamento) : $now;
        
        DB::beginTransaction();
        try {
            // Atualizar negociação
            $valorTotal = (float) $negociacao->valor_total;
            $valorPagoAnterior = (float) ($negociacao->valor_pago ?? 0);
            $novoValorPago = $valorPagoAnterior + $valorPago;
            $saldoRestante = max(0, $valorTotal - $novoValorPago);
            
            // Se o pagamento completar o valor total, marcar como fechado
            $statusNegociacao = 'andamento';
            if ($saldoRestante <= 0.01) { // Tolerância de 1 centavo
                $statusNegociacao = 'fechado';
            }
            
            DB::table('negociacoes')
                ->where('id', $negociacaoId)
                ->update([
                    'valor_pago' => $novoValorPago,
                    'saldo_restante' => $saldoRestante,
                    'status' => $statusNegociacao,
                    'data_fechamento' => $statusNegociacao === 'fechado' ? $now : null,
                    'data_atualizacao' => $now,
                ]);
            
            // Se o pagamento completar, atualizar o lead para "aprovada" (fechamento)
            // Isso move automaticamente o lead da coluna "negociacao" para "aprovada" no Kanban
            if ($statusNegociacao === 'fechado') {
                DB::table('leads')
                    ->where('id', $negociacao->lead_id)
                    ->update([
                        'etapa' => 'fechamento', // Aprovada - aguardando aprovação do coordenador/admin
                        'status' => 'aprovada',
                        'data_atualizacao' => $now,
                    ]);
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Pagamento confirmado com sucesso!',
                'data' => [
                    'negociacao_id' => $negociacaoId,
                    'lead_id' => $negociacao->lead_id,
                    'valor_pago' => $novoValorPago,
                    'saldo_restante' => $saldoRestante,
                    'status' => $statusNegociacao,
                    'lead_atualizado' => $statusNegociacao === 'fechado',
                ]
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Erro ao confirmar pagamento: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Webhook do Pagar.me (quando integrado)
     * Este método recebe notificações do Pagar.me quando um pagamento é confirmado
     */
    public function webhookPagarMe(Request $request)
    {
        // TODO: Implementar validação de assinatura do webhook do Pagar.me
        // TODO: Processar evento do Pagar.me e chamar confirmarPagamento
        
        $eventType = $request->input('event');
        $transactionId = $request->input('transaction.id');
        
        // Buscar negociação pelo transaction_id (quando implementado)
        // Por enquanto, retornar sucesso
        return response()->json([
            'success' => true,
            'message' => 'Webhook recebido'
        ]);
    }
}

