<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Negociacao extends Model
{
    use HasFactory;

    /**
     * Nome da tabela no banco de dados
     */
    protected $table = 'negociacoes';

    /**
     * Desabilitar timestamps automáticos do Laravel
     */
    public $timestamps = false;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'lead_id',
        'especialista_id',
        'valor_total',
        'valor_entrada',
        'num_parcelas',
        'valor_parcela',
        'forma_pagamento',
        'status',
        'data_criacao',
        'data_aprovacao',
        'aprovado_por',
        'observacoes',
        'voucher_id',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'data_criacao' => 'datetime',
        'data_aprovacao' => 'datetime',
        'valor_total' => 'decimal:2',
        'valor_entrada' => 'decimal:2',
        'valor_parcela' => 'decimal:2',
    ];

    /**
     * Relacionamento com Lead
     */
    public function lead()
    {
        return $this->belongsTo(Lead::class, 'lead_id');
    }

    /**
     * Relacionamento com Especialista (Usuario)
     */
    public function especialista()
    {
        return $this->belongsTo(Usuario::class, 'especialista_id');
    }

    /**
     * Relacionamento com Aprovador (Usuario)
     */
    public function aprovador()
    {
        return $this->belongsTo(Usuario::class, 'aprovado_por');
    }

    /**
     * Verifica se a negociação está aprovada
     */
    public function isAprovada(): bool
    {
        return in_array($this->status, ['aprovada', 'finalizada']);
    }

    /**
     * Verifica se a negociação está pendente
     */
    public function isPendente(): bool
    {
        return $this->status === 'pendente';
    }

    /**
     * Scope para negociações aprovadas
     */
    public function scopeAprovadas($query)
    {
        return $query->whereIn('status', ['aprovada', 'finalizada']);
    }

    /**
     * Scope para negociações pendentes
     */
    public function scopePendentes($query)
    {
        return $query->where('status', 'pendente');
    }
}

