<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Venda extends Model
{
    use HasFactory;

    /**
     * Nome da tabela no banco de dados
     */
    protected $table = 'vendas';

    /**
     * Desabilitar timestamps automáticos do Laravel
     */
    public $timestamps = false;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'lead_id',
        'aluno_id',
        'especialista_id',
        'valor_total',
        'valor_entrada',
        'num_parcelas',
        'valor_parcela',
        'forma_pagamento',
        'status',
        'data_venda',
        'data_aprovacao',
        'aprovado_por',
        'observacoes',
        'contrato_id',
        'voucher_id',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'data_venda' => 'datetime',
        'data_aprovacao' => 'datetime',
        'valor_total' => 'decimal:2',
        'valor_entrada' => 'decimal:2',
        'valor_parcela' => 'decimal:2',
    ];

    /**
     * Relacionamento com Lead
     */
    public function lead()
    {
        return $this->belongsTo(Lead::class, 'lead_id');
    }

    /**
     * Relacionamento com Aluno
     */
    public function aluno()
    {
        return $this->belongsTo(Aluno::class, 'aluno_id');
    }

    /**
     * Relacionamento com Especialista
     */
    public function especialista()
    {
        return $this->belongsTo(Especialista::class, 'especialista_id');
    }

    /**
     * Relacionamento com Aprovador (Usuario)
     */
    public function aprovador()
    {
        return $this->belongsTo(Usuario::class, 'aprovado_por');
    }

    /**
     * Relacionamento com Contrato
     */
    public function contrato()
    {
        return $this->belongsTo(Contrato::class, 'contrato_id');
    }

    /**
     * Relacionamento com Voucher
     */
    public function voucher()
    {
        return $this->belongsTo(Voucher::class, 'voucher_id');
    }

    /**
     * Verifica se a venda está aprovada
     */
    public function isAprovada(): bool
    {
        return $this->status === 'aprovada';
    }

    /**
     * Verifica se a venda está pendente
     */
    public function isPendente(): bool
    {
        return $this->status === 'pendente';
    }

    /**
     * Scope para vendas aprovadas
     */
    public function scopeAprovadas($query)
    {
        return $query->where('status', 'aprovada');
    }

    /**
     * Scope para vendas pendentes
     */
    public function scopePendentes($query)
    {
        return $query->where('status', 'pendente');
    }
}

