<?php

namespace App\Services;

use App\Models\Lead;
use App\Models\Especialista;
use App\Models\Equipe;
use Illuminate\Support\Facades\DB;

class LeadDistributionService
{
    /**
     * Distribui um lead automaticamente para o especialista com menos leads ativos
     */
    public function distribuirAutomaticamente(Lead $lead, ?int $equipeId = null): bool
    {
        try {
            DB::beginTransaction();

            // Buscar especialista disponível
            $query = Especialista::where('status', 'ativo');

            if ($equipeId) {
                $query->where('equipe_id', $equipeId);
            }

            // Contar leads ativos por especialista
            $especialista = $query->withCount([
                'leads as leads_ativos_count' => function ($q) {
                    $q->whereIn('status', ['novo', 'em_contato', 'negociacao', 'proposta_enviada']);
                }
            ])
            ->orderBy('leads_ativos_count', 'asc')
            ->first();

            if (!$especialista) {
                DB::rollBack();
                return false;
            }

            // Atribuir lead ao especialista
            $lead->update([
                'especialista_id' => $especialista->id,
                'equipe_id' => $especialista->equipe_id,
                'data_distribuicao' => now(),
            ]);

            DB::commit();
            return true;

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Erro ao distribuir lead automaticamente: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Distribui múltiplos leads de forma balanceada
     */
    public function distribuirEmLote(array $leadIds, ?int $equipeId = null): array
    {
        $resultados = [
            'sucesso' => 0,
            'falha' => 0,
            'erros' => [],
        ];

        foreach ($leadIds as $leadId) {
            $lead = Lead::find($leadId);

            if (!$lead) {
                $resultados['falha']++;
                $resultados['erros'][] = "Lead #{$leadId} não encontrado";
                continue;
            }

            if ($this->distribuirAutomaticamente($lead, $equipeId)) {
                $resultados['sucesso']++;
            } else {
                $resultados['falha']++;
                $resultados['erros'][] = "Erro ao distribuir lead #{$leadId}";
            }
        }

        return $resultados;
    }

    /**
     * Redistribui leads de um especialista para outro
     */
    public function redistribuir(int $especialistaOrigemId, int $especialistaDestinoId): array
    {
        try {
            DB::beginTransaction();

            $especialistaDestino = Especialista::findOrFail($especialistaDestinoId);

            $leads = Lead::where('especialista_id', $especialistaOrigemId)
                ->whereIn('status', ['novo', 'em_contato', 'negociacao', 'proposta_enviada'])
                ->get();

            $total = $leads->count();

            foreach ($leads as $lead) {
                $lead->update([
                    'especialista_id' => $especialistaDestino->id,
                    'equipe_id' => $especialistaDestino->equipe_id,
                    'data_distribuicao' => now(),
                ]);
            }

            DB::commit();

            return [
                'success' => true,
                'total' => $total,
                'message' => "{$total} lead(s) redistribuído(s) com sucesso!",
            ];

        } catch (\Exception $e) {
            DB::rollBack();
            return [
                'success' => false,
                'message' => 'Erro ao redistribuir leads: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Calcula estatísticas de distribuição
     */
    public function getEstatisticasDistribuicao(?int $equipeId = null): array
    {
        $query = Especialista::where('status', 'ativo');

        if ($equipeId) {
            $query->where('equipe_id', $equipeId);
        }

        $especialistas = $query->with('usuario')->withCount([
            'leads as leads_ativos' => function ($q) {
                $q->whereIn('status', ['novo', 'em_contato', 'negociacao', 'proposta_enviada']);
            },
            'leads as leads_novos' => function ($q) {
                $q->where('status', 'novo');
            },
        ])->get();

        $total_leads = $especialistas->sum('leads_ativos');
        $total_especialistas = $especialistas->count();
        $media_leads = $total_especialistas > 0 ? $total_leads / $total_especialistas : 0;

        return [
            'total_leads' => $total_leads,
            'total_especialistas' => $total_especialistas,
            'media_leads' => round($media_leads, 2),
            'especialistas' => $especialistas->map(function ($esp) use ($media_leads) {
                return [
                    'id' => $esp->id,
                    'nome' => $esp->usuario->nome,
                    'leads_ativos' => $esp->leads_ativos,
                    'leads_novos' => $esp->leads_novos,
                    'diferenca_media' => $esp->leads_ativos - $media_leads,
                    'percentual' => $media_leads > 0 ? ($esp->leads_ativos / $media_leads) * 100 : 0,
                ];
            }),
        ];
    }
}

