<?php
if (!isset($_SESSION['especialista_id'])) {
    header('Location: ../login.php');
    exit();
}

// Buscar leads e negociações do especialista
try {
    // Debug: Log do ID do especialista
    error_log("ATENDIMENTOS: Buscando leads para especialista_id = " . ($_SESSION['especialista_id'] ?? 'NÃO DEFINIDO'));
    
    // Buscar leads e negociações - versão NOVA e CORRIGIDA
    $leads_db = [];
    
    // 1. Buscar TODOS os leads atribuídos a este especialista (qualquer status)
    $stmt_meus_leads = $pdo->prepare("
        SELECT 
            id,
            nome,
            email,
            telefone,
            profissao,
            status,
            especialista_id,
            NULL as valor,
            data_cadastro,
            data_distribuicao,
            data_atualizacao,
            observacoes,
            horario_preferencia,
            area_interesse,
            data_agendamento,
            horario_agendamento,
            'lead' as tipo,
            1 as distribuido_para_mim
        FROM leads 
        WHERE especialista_id = ?
        ORDER BY 
            CASE 
                WHEN status = 'aguardando' THEN 1
                WHEN status = 'contatado' THEN 2
                WHEN status = 'negociacao' THEN 3
                WHEN status = 'aprovada' THEN 4
                ELSE 5
            END,
            data_atualizacao DESC,
            data_cadastro DESC
    ");
    $stmt_meus_leads->execute([$_SESSION['especialista_id']]);
    $meus_leads = $stmt_meus_leads->fetchAll(PDO::FETCH_ASSOC);
    
    // Debug: Log da quantidade de leads encontrados
    error_log("ATENDIMENTOS: Encontrados " . count($meus_leads) . " leads para este especialista");
    if (count($meus_leads) > 0) {
        error_log("ATENDIMENTOS: Leads encontrados: " . json_encode($meus_leads));
    }
    
    $leads_db = array_merge($leads_db, $meus_leads);
    
    // 2. Buscar leads NÃO atribuídos (disponíveis para todos)
    $stmt_leads_disponiveis = $pdo->prepare("
        SELECT 
            id,
            nome,
            email,
            telefone,
            profissao,
            status,
            especialista_id,
            NULL as valor,
            data_cadastro,
            data_distribuicao,
            observacoes,
            horario_preferencia,
            area_interesse,
            data_agendamento,
            horario_agendamento,
            'lead' as tipo,
            0 as distribuido_para_mim
        FROM leads 
        WHERE especialista_id IS NULL 
          AND status = 'aguardando'
        ORDER BY data_cadastro DESC
    ");
    $stmt_leads_disponiveis->execute();
    $leads_disponiveis = $stmt_leads_disponiveis->fetchAll(PDO::FETCH_ASSOC);
    
    $leads_db = array_merge($leads_db, $leads_disponiveis);
    
    // 3. Buscar negociações do especialista atual
    // TEMPORARIAMENTE DESABILITADO - Campos da tabela negociacoes precisam ser corrigidos
    /*
    $stmt_negociacoes = $pdo->prepare("
        SELECT 
            id,
            nome_cliente as nome,
            email_cliente as email,
            telefone_cliente as telefone,
            '' as profissao,
            status,
            valor_proposta as valor,
            data_criacao as data_cadastro,
            observacoes,
            'negociacao' as tipo
        FROM negociacoes 
        WHERE especialista_id = ?
    ");
    
    try {
        $stmt_negociacoes->execute([$_SESSION['especialista_id']]);
        $negociacoes = $stmt_negociacoes->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        $negociacoes = []; // Array vazio em caso de erro
    }
    
    $leads_db = array_merge($leads_db, $negociacoes);
    */
    
    // Por enquanto, negociações vazias
    error_log("ATENDIMENTOS: Busca de negociações desabilitada temporariamente");
    
    // Ordenar por data
    usort($leads_db, function($a, $b) {
        return strtotime($b['data_cadastro']) - strtotime($a['data_cadastro']);
    });
    
    
    // Se não houver dados, criar array vazio
    if (empty($leads_db)) {
        $leads_db = [];
    }
} catch (Exception $e) {
    $leads_db = [];
    error_log("ERRO ATENDIMENTOS: " . $e->getMessage());
    error_log("ERRO ATENDIMENTOS: Stack trace: " . $e->getTraceAsString());
    
    // Mostrar erro na tela também para debug
    echo "<div style='background: #f00; color: #fff; padding: 20px; margin: 20px; border-radius: 10px;'>";
    echo "<h3>⚠️ ERRO ao buscar leads:</h3>";
    echo "<p><strong>Mensagem:</strong> " . htmlspecialchars($e->getMessage()) . "</p>";
    echo "<p><strong>Arquivo:</strong> " . $e->getFile() . " (linha " . $e->getLine() . ")</p>";
    echo "<p><a href='debug_leads.php' style='color: #ff0; font-weight: bold;'>→ Clique aqui para abrir o Debug</a></p>";
    echo "</div>";
}
?>

<div class="custom-card">
    <div class="card-body">
        <div class="kanban-container">
            <div class="kanban-board" id="kanban-board">
                <!-- Colunas serão geradas via JavaScript -->
            </div>
        </div>
    </div>
</div>

<!-- Modal para Nova Negociação Rápida -->
<div class="modal fade" id="novaNegociacaoModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header bg-success text-white">
                <h5 class="modal-title">
                    <i class="fas fa-plus-circle me-2"></i>
                    Nova Negociação Rápida
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="formNovaNegociacao">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <div class="form-floating">
                                <input type="text" class="form-control" id="nomeCliente" name="nome_cliente" required>
                                <label for="nomeCliente">Nome do Cliente *</label>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-floating">
                                <input type="email" class="form-control" id="emailCliente" name="email_cliente" required>
                                <label for="emailCliente">Email *</label>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-floating">
                                <input type="text" class="form-control" id="telefoneCliente" name="telefone_cliente">
                                <label for="telefoneCliente">Telefone</label>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-floating">
                                <input type="number" class="form-control" id="valorProposta" name="valor_proposta" step="0.01">
                                <label for="valorProposta">Valor da Proposta</label>
                            </div>
                        </div>
                        <div class="col-12">
                            <div class="form-floating">
                                <textarea class="form-control" id="observacoes" name="observacoes" style="height: 100px;"></textarea>
                                <label for="observacoes">Observações</label>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <i class="fas fa-times me-2"></i>Cancelar
                </button>
                <button type="button" class="btn btn-success" onclick="salvarNovaNegociacao()">
                    <i class="fas fa-save me-2"></i>Criar Negociação
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Modal Detalhes da Negociação -->
<div class="modal fade" id="detalhesNegociacaoModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="fas fa-eye me-2"></i>
                    Detalhes da Negociação
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="detalhesNegociacaoContent">
                <!-- Conteúdo carregado via JavaScript -->
            </div>
        </div>
    </div>
</div>

<!-- Modal de Detalhes do Lead/Negociação (Editável) -->
<div class="details-modal" id="detailsModal">
    <div class="modal-content">
        <div class="modal-header">
            <h3 class="modal-title" id="modalTitle">Detalhes</h3>
            <button class="modal-close" onclick="closeDetailsModal()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <form class="modal-form" id="detailsForm">
            <input type="hidden" id="itemId" name="id">
            <input type="hidden" id="itemTipo" name="tipo">
            
            <div class="form-group">
                <label for="editNome">Nome:</label>
                <input type="text" id="editNome" name="nome" required>
            </div>
            
            <div class="form-group">
                <label for="editEmail">Email:</label>
                <input type="email" id="editEmail" name="email" required>
            </div>
            
            <div class="form-group">
                <label for="editTelefone">Telefone:</label>
                <input type="text" id="editTelefone" name="telefone">
            </div>
            
            <div class="form-group">
                <label for="editProfissao">Profissão:</label>
                <input type="text" id="editProfissao" name="profissao">
            </div>
            
            <div class="form-group">
                <label for="editHorarioPreferencia">
                    <i class="fas fa-calendar-star me-1"></i>Horário de Preferência (do Lead):
                </label>
                <input type="text" id="editHorarioPreferencia" name="horario_preferencia" class="form-control" readonly>
                <small class="text-muted">Data/hora escolhida pelo lead no formulário de captura</small>
            </div>
            
            <div class="row">
                <div class="col-md-6">
                    <div class="form-group">
                        <label for="editDataAgendamento">
                            <i class="fas fa-calendar-check me-1"></i>Data de Agendamento:
                        </label>
                        <input type="date" id="editDataAgendamento" name="data_agendamento" class="form-control">
                        <small class="text-muted">Reagendar para outra data</small>
                    </div>
                </div>
                <div class="col-md-6">
                    <div class="form-group">
                        <label for="editHorarioAgendamento">
                            <i class="fas fa-clock me-1"></i>Horário:
                        </label>
                        <input type="time" id="editHorarioAgendamento" name="horario_agendamento" class="form-control">
                        <small class="text-muted">Reagendar para outro horário</small>
                    </div>
                </div>
            </div>
            
            <div class="form-group" id="areaInteresseGroup">
                <label for="editAreaInteresse">
                    <i class="fas fa-bullseye me-1"></i>Área de Interesse:
                </label>
                <input type="text" id="editAreaInteresse" name="area_interesse" class="form-control" readonly>
                <small class="text-muted">Informação fornecida pelo lead</small>
            </div>
            
            <div class="form-group" id="valorGroup">
                <label for="editValor">Valor (R$):</label>
                <input type="number" id="editValor" name="valor" step="0.01" min="0">
            </div>
            
            <div class="form-group">
                <label for="editStatus">Status:</label>
                <select id="editStatus" name="status" class="form-control">
                    <option value="aguardando">Aguardando</option>
                    <option value="contatado">Contatado</option>
                    <option value="negociacao">Em Negociação</option>
                    <option value="aprovada">Aprovada</option>
                    <option value="rejeitada">Rejeitada</option>
                </select>
            </div>
            
            <div class="form-group" id="observacoesGroup">
                <label for="editObservacoes">Observações:</label>
                <textarea id="editObservacoes" name="observacoes" rows="3" class="form-control"></textarea>
            </div>
        </form>
        
        <div class="modal-actions">
            <button type="button" class="btn-modal btn-cancel" onclick="closeDetailsModal()">
                Cancelar
            </button>
            <button type="button" class="btn-modal btn-save" onclick="saveDetails()">
                Salvar Alterações
            </button>
        </div>
    </div>
</div>

<!-- Modal de Motivo da Perda (Obrigatório) -->
<div class="details-modal" id="lossModal">
    <div class="modal-content">
        <div class="modal-header">
            <h3 class="modal-title">Motivo da Perda</h3>
            <button class="modal-close" onclick="cancelLoss()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <div class="modal-body">
            <div class="alert alert-warning">
                <i class="fas fa-exclamation-triangle me-2"></i>
                <strong>Atenção!</strong> Para marcar como perdida, é obrigatório informar o motivo.
            </div>
            
            <form id="lossForm">
                <input type="hidden" id="lossItemId" name="id">
                <input type="hidden" id="lossItemTipo" name="tipo">
                <input type="hidden" name="status" value="rejeitada">
                
                <div class="form-group">
                    <label for="lossObservacoes">Motivo da perda: <span class="text-danger">*</span></label>
                    <textarea id="lossObservacoes" name="observacoes" rows="4" 
                              placeholder="Descreva o motivo pelo qual essa negociação/lead foi perdida..." 
                              required></textarea>
                    <small class="text-muted">Este campo é obrigatório e ajudará na análise de perdas futuras.</small>
                </div>
            </form>
        </div>
        
        <div class="modal-actions">
            <button type="button" class="btn-modal btn-cancel" onclick="cancelLoss()">
                Cancelar
            </button>
            <button type="button" class="btn-modal btn-save btn-danger" onclick="confirmLoss()">
                Confirmar Perda
            </button>
        </div>
    </div>
</div>

<!-- Modal para Nova Negociação -->
<div class="details-modal" id="negociacaoModal">
    <div class="modal-content">
        <div class="modal-header">
            <h3 class="modal-title">
                <i class="fas fa-handshake me-2"></i>Nova Negociação para Lead
            </h3>
            <button class="modal-close" onclick="closeNegociacaoModal()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <div class="modal-body">
            <div id="negociacaoModalContent">
                <div class="text-center py-4">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Carregando...</span>
                    </div>
                    <p class="mt-2 text-muted">Carregando dados do lead...</p>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Modal de Link de Pagamento -->
<div class="details-modal" id="linkPagamentoModal">
    <div class="modal-content">
        <div class="modal-header bg-success text-white">
            <h3 class="modal-title">
                <i class="fas fa-check-circle me-2"></i>Negociação Criada com Sucesso!
            </h3>
            <button class="modal-close" onclick="closeLinkPagamentoModal()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <div class="modal-body">
            <div class="alert alert-success">
                <i class="fas fa-thumbs-up me-2"></i>
                <strong>Vínculo criado!</strong> O aluno foi vinculado ao lead e a negociação está registrada no sistema.
            </div>
            
            <div id="linkPagamentoContent">
                <!-- Conteúdo será inserido via JavaScript -->
            </div>
            
            <div class="text-center mt-4">
                <p class="text-muted">Próximos passos:</p>
                <ol class="text-start">
                    <li>Copie o link de pagamento acima</li>
                    <li>Envie para o cliente via WhatsApp ou Email</li>
                    <li>Aguarde a confirmação do pagamento</li>
                    <li>Quando o pagamento for confirmado, o lead irá automaticamente para "Aprovada"</li>
                </ol>
            </div>
        </div>
        
        <div class="modal-actions">
            <button type="button" class="btn-modal btn-save" onclick="copiarLinkPagamento()">
                <i class="fas fa-copy me-2"></i>Copiar Link
            </button>
            <button type="button" class="btn-modal btn-primary" onclick="enviarViaWhatsApp()">
                <i class="fab fa-whatsapp me-2"></i>Enviar via WhatsApp
            </button>
            <button type="button" class="btn-modal btn-cancel" onclick="closeLinkPagamentoModal()">
                Fechar
            </button>
        </div>
    </div>
</div>

<script>
// Dados das negociações vindos do PHP
const NEGOCIACOES_DATA = <?php echo json_encode($leads_db, JSON_UNESCAPED_UNICODE); ?>;

// DEBUG: Verificar dados recebidos
console.log('═══════════════════════════════════════════════');
console.log('🔍 DEBUG KANBAN - Dados Recebidos do PHP');
console.log('═══════════════════════════════════════════════');
console.log('Total de leads/negociações:', NEGOCIACOES_DATA.length);
console.log('Dados completos:', NEGOCIACOES_DATA);

// Verificar quantos são leads vs negociações
const leads = NEGOCIACOES_DATA.filter(item => item.tipo === 'lead');
const negociacoes = NEGOCIACOES_DATA.filter(item => item.tipo === 'negociacao');
console.log('Leads:', leads.length);
console.log('Negociações:', negociacoes.length);

// Verificar status dos leads
const porStatus = {};
leads.forEach(lead => {
    const status = lead.status || 'sem_status';
    porStatus[status] = (porStatus[status] || 0) + 1;
});
console.log('Leads por status:', porStatus);
console.log('═══════════════════════════════════════════════');

// Configuração das colunas do kanban
const kanbanConfig = {
    'aguardando': {
        label: 'Aguardando Contato',
        icon: 'fas fa-clock',
        color: 'linear-gradient(90deg, #ff9800, #ffc107)'
    },
    'contatado': {
        label: 'Contatado',
        icon: 'fas fa-phone',
        color: 'linear-gradient(90deg, #2196f3, #1e88e5)'
    },
    'negociacao': {
        label: 'Em Negociação',
        icon: 'fas fa-handshake',
        color: 'linear-gradient(90deg, #9c27b0, #8e24aa)'
    },
    'aprovada': {
        label: 'Aprovada',
        icon: 'fas fa-check-circle',
        color: 'linear-gradient(90deg, #4caf50, #43a047)'
    },
    'rejeitada': {
        label: 'Perdida',
        icon: 'fas fa-times-circle',
        color: 'linear-gradient(90deg, #f44336, #e53935)'
    }
};

// Estado para drag and drop
let draggedCard = null;

// LocalStorage para persistir estado
const LS_KEY = 'especialistaKanbanState';

function saveState() {
    const state = { cardsMap: {} };
    document.querySelectorAll('.kanban-column').forEach(col => {
        col.querySelectorAll('.kanban-card').forEach(card => {
            state.cardsMap[card.dataset.id] = col.dataset.status;
        });
    });
    localStorage.setItem(LS_KEY, JSON.stringify(state));
}

function loadState() {
    try {
        return JSON.parse(localStorage.getItem(LS_KEY)) || { cardsMap: {} };
    } catch {
        return { cardsMap: {} };
    }
}

// Organizar negociações por status
function organizeNegociacoesByStatus() {
    console.log('🔄 Organizando leads por status...');
    
    const organized = {};
    Object.keys(kanbanConfig).forEach(status => {
        organized[status] = [];
    });
    
    // Aplicar estado salvo se existir
    const savedState = loadState();
    console.log('💾 Estado salvo (cache):', savedState);
    
    NEGOCIACOES_DATA.forEach((negociacao, index) => {
        const cardId = `${negociacao.tipo || 'neg'}-${negociacao.id}`;
        let targetStatus = negociacao.status || 'aguardando';
        
        console.log(`\n📝 Processando item ${index + 1}/${NEGOCIACOES_DATA.length}:`);
        console.log('  - ID:', negociacao.id);
        console.log('  - Nome:', negociacao.nome);
        console.log('  - Tipo:', negociacao.tipo);
        console.log('  - Status original:', negociacao.status);
        console.log('  - Distribuído para mim?', negociacao.distribuido_para_mim);
        
        // Mapear status específicos de leads para colunas do kanban
        if (negociacao.tipo === 'lead') {
            // Mapear status de leads para status do kanban
            if (negociacao.status === 'perdida' || negociacao.status === 'rejeitada') {
                targetStatus = 'rejeitada';
            } else if (negociacao.status === 'aguardando_coordenador') {
                targetStatus = 'aguardando';
            } else if (negociacao.status === 'em_negociacao') {
                targetStatus = 'negociacao';
            } else {
                // Para os demais status (aguardando, contatado, negociacao, aprovada), manter o status original
                targetStatus = negociacao.status;
            }
        }
        
        // Se o lead foi recém-distribuído (tem data_distribuicao recente), ignorar cache
        if (negociacao.distribuido_para_mim === 1 && negociacao.data_distribuicao) {
            const dataDistribuicao = new Date(negociacao.data_distribuicao);
            const agora = new Date();
            const diferencaHoras = (agora - dataDistribuicao) / (1000 * 60 * 60);
            
            // Se foi distribuído nas últimas 24 horas, usar status do banco (ignorar cache)
            if (diferencaHoras < 24) {
                console.log(`Lead #${negociacao.id} foi recém-distribuído, usando status do banco: ${targetStatus}`);
            } else {
                // Para leads mais antigos, usar cache se existir
                if (savedState.cardsMap[cardId]) {
                    targetStatus = savedState.cardsMap[cardId];
                }
            }
        } else {
            // Para negociações e outros casos, usar cache se existir
            if (savedState.cardsMap[cardId]) {
                targetStatus = savedState.cardsMap[cardId];
            }
        }
        
        // Garantir que o status existe no kanban
        if (!organized[targetStatus]) {
            console.warn(`Status '${targetStatus}' não existe no kanban, usando 'aguardando'`);
            targetStatus = 'aguardando';
        }
        
        organized[targetStatus].push(negociacao);
        console.log('  ✅ Adicionado à coluna:', targetStatus);
    });
    
    console.log('\n📊 RESULTADO FINAL - Leads organizados por status:');
    Object.keys(organized).forEach(status => {
        console.log(`  - ${status}: ${organized[status].length} leads`);
    });
    console.log('Dados completos:', organized);
    
    return organized;
}

// Renderizar o kanban board
function renderKanbanBoard() {
    console.log('\n🎨 Renderizando Kanban Board...');
    
    const board = document.getElementById('kanban-board');
    if (!board) {
        console.error('❌ Elemento #kanban-board não encontrado!');
        return;
    }
    console.log('✅ Elemento #kanban-board encontrado');
    
    const organizedData = organizeNegociacoesByStatus();
    
    board.innerHTML = '';
    console.log('🧹 Board limpo, iniciando renderização das colunas...');
    
    Object.entries(kanbanConfig).forEach(([status, config]) => {
        const negociacoes = organizedData[status] || [];
        
        console.log(`\n📋 Criando coluna "${config.label}" (${status})`);
        console.log(`   - Quantidade de items: ${negociacoes.length}`);
        
        // Criar coluna
        const column = document.createElement('div');
        column.className = 'kanban-column';
        column.dataset.status = status;
        column.style.setProperty('--column-color', config.color);
        
        // Header da coluna
        const header = document.createElement('div');
        header.className = 'kanban-header';
        header.innerHTML = `
            <div class="kanban-title">
                <i class="${config.icon}"></i>
                <span>${config.label}</span>
            </div>
            <div class="kanban-count">${negociacoes.length}</div>
        `;
        
        // Container dos cards
        const cardsContainer = document.createElement('div');
        cardsContainer.className = 'kanban-cards';
        
        if (negociacoes.length === 0) {
            cardsContainer.innerHTML = `
                <div class="kanban-empty">
                    <i class="${config.icon}"></i>
                    <p>Nenhuma negociação</p>
                </div>
            `;
        } else {
            negociacoes.forEach(negociacao => {
                const card = createNegociacaoCard(negociacao);
                cardsContainer.appendChild(card);
            });
        }
        
        // Event listeners para drop
        cardsContainer.addEventListener('dragover', e => e.preventDefault());
        cardsContainer.addEventListener('drop', handleDrop);
        
        column.appendChild(header);
        column.appendChild(cardsContainer);
        board.appendChild(column);
        console.log(`   ✅ Coluna "${config.label}" adicionada ao board`);
    });
    
    console.log('\n✅ Kanban Board renderizado com sucesso!');
    console.log('═══════════════════════════════════════════════\n');
    
    updateStats();
}

// Função para escapar HTML
function escapeHtml(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Função para escapar dados para atributos HTML
function escapeAttr(text) {
    if (!text) return '';
    return text.replace(/"/g, '&quot;').replace(/'/g, '&#39;');
}

// Criar card de negociação/lead
function createNegociacaoCard(negociacao) {
    const card = document.createElement('div');
    card.className = 'kanban-card';
    card.draggable = true;
    card.dataset.id = `${negociacao.tipo || 'neg'}-${negociacao.id}`;
    card.dataset.negociacaoId = negociacao.id;
    card.dataset.tipo = negociacao.tipo || 'negociacao';
    
    // Verificar se é um lead distribuído para este especialista
    const isDistribuidoParaMim = negociacao.tipo === 'lead' && 
                                  negociacao.distribuido_para_mim === 1 &&
                                  negociacao.especialista_id;
    
    // Adicionar classe especial para leads distribuídos
    if (isDistribuidoParaMim) {
        card.classList.add('lead-distribuido');
    }
    
    const valor = parseFloat(negociacao.valor || 0);
    const valorFormatado = valor > 0 ? 
        `R$ ${valor.toLocaleString('pt-BR', { minimumFractionDigits: 2 })}` : 
        'Valor não informado';
    
    // Formatar data de agendamento - PRIORIZAR horario_preferencia (vem da captura de lead)
    let dataExibicao = 'Sem data';
    let iconeData = 'fas fa-calendar';
    let tipoData = 'cadastro';
    
    console.log('Lead #' + negociacao.id + ' - Dados de data:', {
        horario_preferencia: negociacao.horario_preferencia,
        data_agendamento: negociacao.data_agendamento,
        horario_agendamento: negociacao.horario_agendamento,
        data_cadastro: negociacao.data_cadastro
    });
    
    // PRIORIDADE 1: horario_preferencia (vem da captura de lead)
    if (negociacao.horario_preferencia && negociacao.horario_preferencia !== null && negociacao.horario_preferencia !== '') {
        // Processar horario_preferencia - pode vir em dois formatos:
        // Formato 1: "DD/MM/YYYY às HH:MM" (antigo)
        // Formato 2: "YYYY-MM-DDTHH:MM" (ISO - novo)
        
        let horarioPref = negociacao.horario_preferencia;
        
        // Se for formato ISO (YYYY-MM-DDTHH:MM)
        if (horarioPref.includes('T')) {
            const partes = horarioPref.split('T');
            const data = new Date(partes[0] + 'T00:00:00');
            const hora = partes[1].substring(0, 5); // HH:MM
            dataExibicao = data.toLocaleDateString('pt-BR') + ' às ' + hora;
        } 
        // Se já está no formato "DD/MM/YYYY às HH:MM"
        else {
            dataExibicao = horarioPref;
        }
        
        iconeData = 'fas fa-calendar-check';
        tipoData = 'preferencia';
        console.log('Lead #' + negociacao.id + ' - USANDO HORÁRIO PREFERÊNCIA:', dataExibicao);
    }
    // PRIORIDADE 2: data_agendamento (editado pelo especialista)
    else if (negociacao.data_agendamento && negociacao.data_agendamento !== null && negociacao.data_agendamento !== '') {
        const dataAgend = new Date(negociacao.data_agendamento + 'T00:00:00');
        dataExibicao = dataAgend.toLocaleDateString('pt-BR');
        
        if (negociacao.horario_agendamento && negociacao.horario_agendamento !== null && negociacao.horario_agendamento !== '') {
            dataExibicao += ' às ' + negociacao.horario_agendamento.substring(0, 5);
        }
        
        iconeData = 'fas fa-calendar-check';
        tipoData = 'agendamento';
        console.log('Lead #' + negociacao.id + ' - USANDO AGENDAMENTO EDITADO:', dataExibicao);
    }
    // PRIORIDADE 3: data_cadastro (fallback)
    else if (negociacao.data_cadastro) {
        const dataCad = new Date(negociacao.data_cadastro);
        dataExibicao = dataCad.toLocaleDateString('pt-BR');
        iconeData = 'fas fa-calendar-plus';
        tipoData = 'cadastro';
        console.log('Lead #' + negociacao.id + ' - USANDO CADASTRO:', dataExibicao);
    }
    
    // Formatar telefone para WhatsApp (remover caracteres especiais)
    let whatsappLink = '#';
    let telefoneDisplay = negociacao.telefone || 'Sem telefone';
    
    if (negociacao.telefone) {
        // Remover tudo exceto números
        const telefoneNumeros = negociacao.telefone.replace(/\D/g, '');
        
        // Se tiver 11 dígitos ou mais, criar link do WhatsApp
        if (telefoneNumeros.length >= 10) {
            whatsappLink = `https://wa.me/55${telefoneNumeros}`;
            console.log('WhatsApp link para Lead #' + negociacao.id + ':', whatsappLink);
        }
    }
    
    card.innerHTML = `
        <div class="negociacao-header">
            <span class="negociacao-id">#${String(negociacao.id).padStart(4, '0')}</span>
        </div>
        
        <button class="view-details-btn" 
                onclick="event.stopPropagation(); openDetailsModal(${negociacao.id})" 
                draggable="false"
                onmousedown="event.stopPropagation()"
                ondragstart="event.preventDefault(); event.stopPropagation(); return false;"
                title="Ver detalhes completos">
            <i class="fas fa-eye"></i>
        </button>
        
        <h4 class="negociacao-title">${escapeHtml(negociacao.nome || 'Cliente sem nome')}</h4>
        
        <div class="negociacao-meta">
            <div class="meta-item">
                <a href="${whatsappLink}" 
                   target="_blank" 
                   class="whatsapp-link" 
                   draggable="false"
                   onclick="event.stopPropagation()"
                   onmousedown="event.stopPropagation()"
                   ondragstart="event.preventDefault(); event.stopPropagation(); return false;"
                   title="Abrir conversa no WhatsApp">
                    <i class="fab fa-whatsapp"></i>
                    <span>${escapeHtml(telefoneDisplay)}</span>
                </a>
            </div>
            <div class="meta-item">
                <i class="${iconeData}"></i>
                <span>${dataExibicao}</span>
            </div>
        </div>
    `;
    
    // Event listeners
    card.addEventListener('dragstart', handleDragStart);
    card.addEventListener('dragend', handleDragEnd);
    
    return card;
}

// Handlers para drag and drop
function handleDragStart(e) {
    // Não permitir drag se o clique foi em um elemento interativo
    const target = e.target;
    
    // Verificar se o clique foi em um link, botão ou elemento dentro deles
    if (target.closest('a') || target.closest('button') || target.tagName === 'A' || target.tagName === 'BUTTON') {
        e.preventDefault();
        return false;
    }
    
    // Se o elemento clicado não for o card, encontrar o card pai
    const card = target.closest('.kanban-card');
    if (!card) {
        e.preventDefault();
        return false;
    }
    
    draggedCard = card;
    card.classList.add('dragging');
    e.dataTransfer.effectAllowed = 'move';
    e.dataTransfer.setData('text/html', card.outerHTML);
}

function handleDragEnd(e) {
    // Encontrar o card que está sendo arrastado
    const card = e.target.closest('.kanban-card') || draggedCard;
    if (card) {
        card.classList.remove('dragging');
    }
    draggedCard = null;
}

function handleDrop(e) {
    e.preventDefault();
    
    if (!draggedCard) return;
    
    const targetColumn = e.target.closest('.kanban-column');
    const targetCards = e.target.closest('.kanban-cards');
    
    if (targetColumn && targetCards && draggedCard.parentElement !== targetCards) {
        const newStatus = targetColumn.dataset.status;
        const negociacaoId = draggedCard.dataset.negociacaoId;
        const cardTipo = draggedCard.dataset.tipo;
        
        // Se for um LEAD movido para "negociacao", abrir modal de nova negociação
        if (cardTipo === 'lead' && newStatus === 'negociacao') {
            // Armazenar dados temporários
            window.tempDropData = {
                draggedCard: draggedCard,
                originalParent: draggedCard.parentElement,
                targetCards: targetCards,
                negociacaoId: negociacaoId,
                newStatus: newStatus
            };
            
            // Abrir modal de nova negociação
            openNegociacaoModal(negociacaoId);
            return;
        }
        
        // Se for movido para "rejeitada", abrir modal obrigatório
        if (newStatus === 'rejeitada') {
            // Armazenar dados temporários para o caso de cancelamento
            window.tempDropData = {
                draggedCard: draggedCard,
                originalParent: draggedCard.parentElement,
                targetCards: targetCards,
                negociacaoId: negociacaoId,
                newStatus: newStatus
            };
            
            // Abrir modal obrigatório de perda
            openLossModal(negociacaoId);
            return;
        }
        
        // Para outros status, processar normalmente
        processCardMove(draggedCard, targetCards, negociacaoId, newStatus);
    }
}

// Função para processar movimento normal do card
function processCardMove(card, targetCards, negociacaoId, newStatus) {
    // Mover o card
    targetCards.appendChild(card);
    
    // Atualizar contador
    updateColumnCounts();
    
    // Salvar estado
    saveState();
    
    // Atualizar no banco de dados
    updateNegociacaoStatus(negociacaoId, newStatus);
    
    // Se movido para aprovada, mostrar mensagem especial
    if (newStatus === 'aprovada') {
        showSuccessMessage('Negociação aprovada! Parabéns pelo fechamento! 🎉');
    }
}

// Atualizar status no banco de dados
async function updateNegociacaoStatus(negociacaoId, newStatus) {
    try {
        // Buscar o tipo do card
        const card = document.querySelector(`[data-negociacao-id="${negociacaoId}"]`);
        const tipo = card ? card.dataset.tipo : 'negociacao';
        
        const formData = new FormData();
        formData.append('action', 'update_status');
        formData.append('negociacao_id', negociacaoId);
        formData.append('new_status', newStatus);
        formData.append('tipo', tipo);
        
        const response = await fetch('ajax/update_status.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            // Atualizar contadores após sucesso na base de dados
            updateColumnCounts();
        } else {
            console.error('Erro ao atualizar status:', result.error);
            showErrorMessage('Erro ao atualizar status da negociação');
        }
    } catch (error) {
        console.error('Erro na requisição:', error);
        showErrorMessage('Erro na comunicação com o servidor');
    }
}

// Atualizar contadores das colunas
function updateColumnCounts() {
    document.querySelectorAll('.kanban-column').forEach(column => {
        const cards = column.querySelectorAll('.kanban-card');
        const count = cards.length;
        const countElement = column.querySelector('.kanban-count');
        const cardsContainer = column.querySelector('.kanban-cards');
        
        // Atualizar contador
        if (countElement) {
            countElement.textContent = count;
        }
        
        // Gerenciar conteúdo da coluna
        if (cardsContainer) {
            const emptyDiv = cardsContainer.querySelector('.kanban-empty');
            
            if (count === 0) {
                // Se não há cards, mostrar mensagem vazia
                if (!emptyDiv) {
                    const status = column.dataset.status;
                    const config = kanbanConfig[status];
                    cardsContainer.innerHTML = `
                        <div class="kanban-empty">
                            <i class="${config.icon}"></i>
                            <p>Nenhuma negociação</p>
                        </div>
                    `;
                }
            } else {
                // Se há cards, remover mensagem vazia
                if (emptyDiv) {
                    emptyDiv.remove();
                }
            }
        }
    });
}

// Atualizar estatísticas (removido - KPIs foram removidas)
function updateStats() {
    // Função mantida para compatibilidade, mas sem funcionalidade
    console.log('Stats updated - KPIs removed');
}

// Mostrar detalhes da negociação/lead
function showNegociacaoDetails(negociacao) {
    const modal = document.getElementById('detalhesNegociacaoModal');
    const content = document.getElementById('detalhesNegociacaoContent');
    
    const valor = parseFloat(negociacao.valor || 0);
    const valorFormatado = valor > 0 ? 
        `R$ ${valor.toLocaleString('pt-BR', { minimumFractionDigits: 2 })}` : 
        'Valor não informado';
    
    const tipoLabel = negociacao.tipo === 'lead' ? 'Lead' : 'Negociação';
    const statusBadgeClass = negociacao.tipo === 'lead' ? 'bg-success' : 'bg-primary';
    
    content.innerHTML = `
        <div class="row">
            <div class="col-md-8">
                <h4>${escapeHtml(negociacao.nome || 'Cliente sem nome')}</h4>
                <p class="text-muted">${tipoLabel} #${String(negociacao.id).padStart(4, '0')}</p>
            </div>
            <div class="col-md-4 text-end">
                <span class="badge ${statusBadgeClass} fs-6">${kanbanConfig[negociacao.status]?.label || 'Status não definido'}</span>
            </div>
        </div>
        
        <div class="row mb-3">
            <div class="col-md-6">
                <strong>Email:</strong><br>
                <span class="text-muted">${escapeHtml(negociacao.email || 'Não informado')}</span>
            </div>
            <div class="col-md-6">
                <strong>Telefone:</strong><br>
                <span class="text-muted">${escapeHtml(negociacao.telefone || 'Não informado')}</span>
            </div>
        </div>
        
        ${negociacao.profissao ? `
        <div class="row mb-3">
            <div class="col-md-6">
                <strong>Profissão:</strong><br>
                <span class="text-muted">${escapeHtml(negociacao.profissao)}</span>
            </div>
            <div class="col-md-6">
                <strong>Tipo:</strong><br>
                <span class="badge ${negociacao.tipo === 'lead' ? 'bg-success' : 'bg-primary'}">${tipoLabel}</span>
            </div>
        </div>
        ` : ''}
        
        <div class="row mb-3">
            <div class="col-md-6">
                <strong>${negociacao.tipo === 'lead' ? 'Status do Lead:' : 'Valor da Proposta:'}</strong><br>
                <span class="${negociacao.tipo === 'lead' ? 'text-success' : 'text-success'} fs-5">${valorFormatado}</span>
            </div>
            <div class="col-md-6">
                <strong>Data de ${negociacao.tipo === 'lead' ? 'Cadastro' : 'Criação'}:</strong><br>
                <span class="text-muted">${negociacao.data_cadastro ? new Date(negociacao.data_cadastro).toLocaleDateString('pt-BR') : 'Não informado'}</span>
            </div>
        </div>
        
        ${negociacao.observacoes ? `
        <div class="mb-3">
            <strong>Observações:</strong><br>
            <div class="border rounded p-3 bg-light">
                ${escapeHtml(negociacao.observacoes)}
            </div>
        </div>
        ` : ''}
        
        <div class="text-end">
            <button class="btn btn-primary" onclick="editarNegociacao(${negociacao.id})">
                <i class="fas fa-edit me-2"></i>Editar
            </button>
        </div>
    `;
    
    const bsModal = new bootstrap.Modal(modal);
    bsModal.show();
}

// Funções utilitárias
function showSuccessMessage(message) {
    const alert = document.createElement('div');
    alert.className = 'alert alert-success alert-dismissible fade show';
    alert.innerHTML = `
        <i class="fas fa-check-circle me-2"></i>
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.querySelector('.modern-content').insertBefore(alert, document.querySelector('.kanban-container'));
    
    setTimeout(() => {
        alert.remove();
    }, 5000);
}

function showErrorMessage(message) {
    const alert = document.createElement('div');
    alert.className = 'alert alert-danger alert-dismissible fade show';
    alert.innerHTML = `
        <i class="fas fa-exclamation-triangle me-2"></i>
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.querySelector('.modern-content').insertBefore(alert, document.querySelector('.kanban-container'));
    
    setTimeout(() => {
        alert.remove();
    }, 5000);
}

function editarNegociacao(id) {
    // Redirecionar para página de edição ou abrir modal de edição
    window.location.href = `index.php?page=nova_negociacao&edit=${id}`;
}

// Funções do modal de perda
async function openLossModal(itemId) {
    // Buscar os dados da negociação/lead pelo ID
    const item = NEGOCIACOES_DATA.find(neg => neg.id == itemId);
    
    if (!item) {
        alert('Item não encontrado');
        return;
    }
    
    const modal = document.getElementById('lossModal');
    
    // Preencher campos
    document.getElementById('lossItemId').value = item.id;
    document.getElementById('lossItemTipo').value = item.tipo || 'negociacao';
    document.getElementById('lossObservacoes').value = item.observacoes || '';
    
    // Mostrar modal
    modal.style.display = 'block';
    document.body.style.overflow = 'hidden';
    
    // Focar no campo de observações
    setTimeout(() => {
        document.getElementById('lossObservacoes').focus();
    }, 100);
}

function closeLossModal() {
    const modal = document.getElementById('lossModal');
    modal.style.display = 'none';
    document.body.style.overflow = 'auto';
    
    // Limpar formulário
    document.getElementById('lossForm').reset();
}

function cancelLoss() {
    // Retornar card para posição original se cancelar
    if (window.tempDropData) {
        const { draggedCard, originalParent } = window.tempDropData;
        originalParent.appendChild(draggedCard);
        updateColumnCounts();
        window.tempDropData = null;
    }
    
    closeLossModal();
}

function confirmLoss() {
    const observacoes = document.getElementById('lossObservacoes').value.trim();
    
    if (!observacoes) {
        alert('O motivo da perda é obrigatório!');
        document.getElementById('lossObservacoes').focus();
        return;
    }
    
    if (!window.tempDropData) {
        alert('Erro: dados temporários não encontrados');
        return;
    }
    
    const { draggedCard, targetCards, negociacaoId } = window.tempDropData;
    
    // Mover o card visualmente
    targetCards.appendChild(draggedCard);
    
    // Atualizar contador
    updateColumnCounts();
    
    // Salvar estado local
    saveState();
    
    // Atualizar observações e status via AJAX (faz tudo de uma vez)
    updateLossObservations(negociacaoId, observacoes);
    
    // Limpar dados temporários
    window.tempDropData = null;
    
    // Fechar modal
    closeLossModal();
    
    // Mostrar mensagem de confirmação
    showSuccessMessage('Lead/Negociação marcada como perdida com motivo registrado.');
}

async function updateLossObservations(itemId, observacoes) {
    // Buscar tipo do item
    const item = NEGOCIACOES_DATA.find(neg => neg.id == itemId);
    if (!item) {
        console.error('Item não encontrado:', itemId);
        showErrorMessage('Item não encontrado');
        return;
    }
    
    const data = {
        id: itemId,
        tipo: item.tipo || 'negociacao',
        observacoes: observacoes
    };
    
    console.log('📤 Salvando lead como perdido:', data);
    
    try {
        const response = await fetch('ajax/update_lead_perdido.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(data)
        });
        
        const result = await response.json();
        
        if (result.success) {
            console.log('✅ Lead marcado como perdido com sucesso');
            
            // Atualizar dados locais
            const localItem = NEGOCIACOES_DATA.find(neg => neg.id == itemId);
            if (localItem) {
                localItem.observacoes = observacoes;
                localItem.status = 'perdida';
            }
            
            showSuccessMessage(result.message || 'Lead marcado como perdido com sucesso');
        } else {
            console.error('❌ Erro ao marcar como perdido:', result.error);
            showErrorMessage(result.error || 'Erro ao salvar');
        }
        
    } catch (error) {
        console.error('❌ Erro na requisição:', error);
        showErrorMessage('Erro na comunicação com o servidor: ' + error.message);
    }
}

// ===== FUNÇÕES PARA MODAL DE NEGOCIAÇÃO =====

// Abrir modal de nova negociação para lead
async function openNegociacaoModal(leadId) {
    try {
        console.log('Abrindo modal de negociação para lead:', leadId);
        
        // Carregar conteúdo do modal
        const response = await fetch(`/portal/especialista/ajax/get_negociacao_modal.php?lead_id=${leadId}`);
        const result = await response.json();
        
        if (result.error) {
            throw new Error(result.error);
        }
        
        // Inserir conteúdo no modal
        document.getElementById('negociacaoModalContent').innerHTML = result.html;
        
        // Inicializar event listeners do modal após inserir HTML
        console.log('HTML inserido, inicializando event listeners...');
        setTimeout(() => {
            inicializarModalNegociacaoAtendimentos();
        }, 100);
        
        // Abrir modal
        const modal = document.getElementById('negociacaoModal');
        modal.style.display = 'flex';
        
    } catch (error) {
        console.error('Erro ao abrir modal de negociação:', error);
        alert('❌ Erro ao carregar modal de negociação: ' + error.message);
        
        // Reverter movimento se houver erro
        if (window.tempDropData) {
            window.tempDropData.originalParent.appendChild(window.tempDropData.draggedCard);
            window.tempDropData = null;
        }
    }
}

// Inicializar modal de negociação (event listeners e cálculos)
function inicializarModalNegociacaoAtendimentos() {
    console.log('🔧 Inicializando modal de negociação...');
    
    const voucherSelect = document.getElementById('voucherModal');
    const formaPagamentoSelect = document.getElementById('formaPagamento');
    const parcelasSelect = document.getElementById('numParcelasModal');
    
    if (!voucherSelect || !formaPagamentoSelect || !parcelasSelect) {
        console.error('❌ Elementos não encontrados:', {
            voucherSelect: !!voucherSelect,
            formaPagamentoSelect: !!formaPagamentoSelect,
            parcelasSelect: !!parcelasSelect
        });
        return;
    }
    
    console.log('✅ Todos os elementos encontrados!');
    
    // Event listener para voucher
    voucherSelect.addEventListener('change', function() {
        console.log('🎟️ Voucher mudou! Valor:', this.value);
        const selectedOption = this.options[this.selectedIndex];
        const desconto = selectedOption.getAttribute('data-desconto');
        console.log('Desconto no data-desconto:', desconto);
        updateValorTotalModalAtendimentos();
    });
    
    // Event listener para forma de pagamento
    formaPagamentoSelect.addEventListener('change', function() {
        console.log('💳 Forma de pagamento mudou:', this.value);
        updateParcelasModalAtendimentos();
    });
    
    // Event listener para parcelas
    parcelasSelect.addEventListener('change', function() {
        console.log('📊 Parcelas mudou:', this.value);
        updateValorTotalModalAtendimentos();
    });
    
    // Inicializar valores
    updateParcelasModalAtendimentos();
    
    console.log('✅ Modal inicializado com sucesso!');
}

// Função para atualizar parcelas baseado na forma de pagamento
function updateParcelasModalAtendimentos() {
    const formaPagamento = document.getElementById('formaPagamento').value;
    const selectParcelas = document.getElementById('numParcelasModal');
    const valorIntegralElement = document.querySelector('#formNegociacaoModal input[name="tipo_valor"]');
    const VALOR_INTEGRAL = parseFloat(valorIntegralElement?.closest('.form-group').querySelector('input[readonly]')?.value.replace(/[^\d,]/g, '').replace(',', '.')) || 5000;
    
    selectParcelas.innerHTML = '';
    
    if (formaPagamento === 'credito') {
        for (let i = 1; i <= 12; i++) {
            const option = document.createElement('option');
            option.value = i;
            option.textContent = `${i}x`;
            if (i > 1) {
                const valorParcela = (VALOR_INTEGRAL * Math.pow(1.0199, i)) / i;
                option.textContent += ` (R$ ${valorParcela.toFixed(2).replace('.', ',')})`;
            }
            selectParcelas.appendChild(option);
        }
    } else if (formaPagamento === 'pix') {
        selectParcelas.innerHTML = '<option value="1">1x (à vista)</option><option value="2">2x (entrada + 30 dias)</option>';
    }
    
    updateValorTotalModalAtendimentos();
}

// Função para atualizar valor total com desconto do voucher
function updateValorTotalModalAtendimentos() {
    console.log('=== updateValorTotalModalAtendimentos CHAMADA ===');
    
    const voucherSelect = document.getElementById('voucherModal');
    const parcelasSelect = document.getElementById('numParcelasModal');
    const formaPagamentoSelect = document.getElementById('formaPagamento');
    const valorTotalSpan = document.getElementById('valorTotalSpanModal');
    const descontoInfo = document.getElementById('descontoInfoModal');
    const descontoPercentual = document.getElementById('descontoPercentualModal');
    const valorOriginal = document.getElementById('valorOriginalModal');
    const valorBox = document.getElementById('valorTotalBoxModal');
    
    // Obter valor integral
    const valorIntegralElement = document.querySelector('#formNegociacaoModal input[name="tipo_valor"]');
    const VALOR_INTEGRAL = parseFloat(valorIntegralElement?.closest('.form-group').querySelector('input[readonly]')?.value.replace(/[^\d,]/g, '').replace(',', '.')) || 5000;
    
    console.log('Valor Integral obtido:', VALOR_INTEGRAL);
    
    let valorTotal = VALOR_INTEGRAL;
    let descontoAplicado = 0;
    
    // Aplicar desconto do voucher
    if (voucherSelect && voucherSelect.value) {
        const selectedOption = voucherSelect.options[voucherSelect.selectedIndex];
        descontoAplicado = parseFloat(selectedOption.getAttribute('data-desconto')) || 0;
        
        console.log('📌 Voucher selecionado:', {
            id: voucherSelect.value,
            desconto: descontoAplicado + '%',
            valorOriginal: VALOR_INTEGRAL
        });
        
        valorTotal = VALOR_INTEGRAL * (1 - descontoAplicado / 100);
        
        console.log('💰 Valor com desconto:', valorTotal);
    }
    
    // Aplicar juros se cartão de crédito parcelado
    const parcelas = parseInt(parcelasSelect?.value || 1);
    const formaPagamento = formaPagamentoSelect?.value || 'credito';
    
    if (formaPagamento === 'credito' && parcelas > 1) {
        valorTotal = valorTotal * Math.pow(1.0199, parcelas);
        console.log('💳 Valor com juros de ' + parcelas + 'x:', valorTotal);
    }
    
    // Atualizar display
    if (valorTotalSpan) {
        valorTotalSpan.textContent = 'R$ ' + valorTotal.toFixed(2).replace('.', ',');
        console.log('✅ Valor atualizado na tela:', valorTotal);
    }
    
    // Mostrar/esconder informações de desconto
    if (descontoAplicado > 0) {
        if (descontoInfo) descontoInfo.style.display = 'block';
        if (descontoPercentual) descontoPercentual.textContent = descontoAplicado + '%';
        if (valorOriginal) valorOriginal.style.display = 'block';
        if (valorBox) {
            valorBox.classList.remove('alert-success');
            valorBox.classList.add('alert-warning');
        }
    } else {
        if (descontoInfo) descontoInfo.style.display = 'none';
        if (valorOriginal) valorOriginal.style.display = 'none';
        if (valorBox) {
            valorBox.classList.remove('alert-warning');
            valorBox.classList.add('alert-success');
        }
    }
}

// Toggle investimento parcial
function toggleInvestimentoParcial() {
    const checkbox = document.getElementById('investimentoParcialCheck');
    const investimentoGroup = document.getElementById('investimentoParcialGroup');
    const valorInvestidoInput = document.getElementById('valorInvestido');
    
    if (checkbox && checkbox.checked) {
        investimentoGroup.style.display = 'block';
        if (valorInvestidoInput) {
            valorInvestidoInput.required = true;
        }
    } else {
        investimentoGroup.style.display = 'none';
        if (valorInvestidoInput) {
            valorInvestidoInput.required = false;
            valorInvestidoInput.value = '';
        }
        // Esconder info de saldo
        const saldoInfo = document.getElementById('saldoRestanteInfo');
        if (saldoInfo) {
            saldoInfo.style.display = 'none';
        }
    }
}

// Calcular saldo restante
function calcularSaldoRestante() {
    const valorInvestidoInput = document.getElementById('valorInvestido');
    const valorTotalSpan = document.getElementById('valorTotalSpanModal');
    const saldoRestanteInfo = document.getElementById('saldoRestanteInfo');
    const saldoRestanteValor = document.getElementById('saldoRestanteValor');
    
    if (!valorInvestidoInput || !valorTotalSpan) return;
    
    const valorInvestido = parseFloat(valorInvestidoInput.value) || 0;
    
    // Obter valor total atual (remover formatação)
    const valorTotalText = valorTotalSpan.textContent.replace('R$', '').replace(/\./g, '').replace(',', '.').trim();
    const valorTotal = parseFloat(valorTotalText) || 0;
    
    const saldoRestante = valorTotal - valorInvestido;
    
    console.log('Cálculo de saldo:', {
        valorTotal: valorTotal,
        valorInvestido: valorInvestido,
        saldoRestante: saldoRestante
    });
    
    if (valorInvestido > 0) {
        if (saldoRestanteInfo) {
            saldoRestanteInfo.style.display = 'block';
        }
        if (saldoRestanteValor) {
            saldoRestanteValor.textContent = 'R$ ' + saldoRestante.toFixed(2).replace('.', ',');
            
            // Mudar cor baseado no saldo
            if (saldoRestante > 0) {
                saldoRestanteInfo.classList.remove('alert-success');
                saldoRestanteInfo.classList.add('alert-info');
            } else if (saldoRestante === 0) {
                saldoRestanteInfo.classList.remove('alert-info');
                saldoRestanteInfo.classList.add('alert-success');
            }
        }
    } else {
        if (saldoRestanteInfo) {
            saldoRestanteInfo.style.display = 'none';
        }
    }
}

// Cancelar vínculo existente
async function cancelarVinculoExistente(vinculoId, negociacaoId) {
    if (!confirm('⚠️ Tem certeza que deseja cancelar este vínculo?\n\nIsso irá:\n- Desvinc ular o lead do aluno\n- Cancelar a negociação associada\n- Retornar o lead ao status anterior\n\nEsta ação não pode ser desfeita.')) {
        return;
    }
    
    try {
        const response = await fetch('/portal/especialista/ajax/cancelar_vinculo.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                vinculo_id: vinculoId,
                negociacao_id: negociacaoId
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            showSuccessMessage('✅ Vínculo cancelado com sucesso!');
            closeNegociacaoModal();
            
            // Recarregar página para atualizar dados
            setTimeout(() => {
                location.reload();
            }, 1000);
        } else {
            throw new Error(result.message);
        }
    } catch (error) {
        console.error('Erro ao cancelar vínculo:', error);
        alert('❌ Erro ao cancelar vínculo: ' + error.message);
    }
}

// Fechar modal de negociação
function closeNegociacaoModal() {
    const modal = document.getElementById('negociacaoModal');
    modal.style.display = 'none';
    
    // Reverter movimento se modal foi fechado sem salvar
    if (window.tempDropData) {
        console.log('Modal fechado sem salvar - revertendo movimento');
        window.tempDropData.originalParent.appendChild(window.tempDropData.draggedCard);
        window.tempDropData = null;
    }
}

// Confirmar criação da negociação
async function confirmNegociacao() {
    const form = document.getElementById('formNegociacaoModal');
    const formData = new FormData(form);
    const leadId = form.dataset.leadId;
    
    // Validar campos obrigatórios
    const alunoId = formData.get('aluno_existente_id');
    if (!alunoId) {
        alert('❌ Por favor, selecione um aluno para vincular ao lead.');
        return;
    }
    
    // Converter FormData para objeto
    const data = {};
    formData.forEach((value, key) => {
        data[key] = value;
    });
    data.lead_id = leadId;
    
    try {
        console.log('Enviando dados da negociação:', data);
        
        const response = await fetch('/portal/especialista/ajax/salvar_negociacao_vinculo.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(data)
        });
        
        const result = await response.json();
        
        if (result.success) {
            console.log('✅ Negociação criada com sucesso:', result.data);
            
            // Atualizar status do lead no array local
            const leadIndex = NEGOCIACOES_DATA.findIndex(item => item.id == leadId);
            if (leadIndex !== -1) {
                NEGOCIACOES_DATA[leadIndex].status = 'negociacao';
                console.log('Status do lead atualizado no array local para: negociacao');
            }
            
            // Processar movimento do card
            if (window.tempDropData) {
                // Mover o card para a coluna de negociação
                window.tempDropData.targetCards.appendChild(window.tempDropData.draggedCard);
                
                // Atualizar data-status do card
                window.tempDropData.draggedCard.dataset.status = 'negociacao';
                
                // Salvar estado no localStorage
                saveState();
                
                // Atualizar contadores
                updateColumnCounts();
                
                window.tempDropData = null;
                
                console.log('✅ Card mantido na coluna "Em Negociação"');
            }
            
            // Fechar modal de negociação
            closeNegociacaoModal();
            
            // Mostrar mensagem de sucesso
            showSuccessMessage('✅ ' + result.message);
            
            // Abrir modal de link de pagamento
            abrirModalLinkPagamento(result.data);
            
        } else {
            throw new Error(result.message);
        }
        
    } catch (error) {
        console.error('Erro ao salvar negociação:', error);
        alert('❌ Erro ao salvar negociação: ' + error.message);
        
        // Reverter movimento se houver erro
        if (window.tempDropData) {
            window.tempDropData.originalParent.appendChild(window.tempDropData.draggedCard);
            window.tempDropData = null;
        }
    }
}

// Variável global para armazenar dados do pagamento
let dadosPagamentoAtual = null;

// Abrir modal de link de pagamento
function abrirModalLinkPagamento(dadosNegociacao) {
    console.log('Abrindo modal de link de pagamento:', dadosNegociacao);
    
    dadosPagamentoAtual = dadosNegociacao;
    
    const modal = document.getElementById('linkPagamentoModal');
    const content = document.getElementById('linkPagamentoContent');
    
    // Usar link retornado pela API do Pagar.me
    const linkPagamento = dadosNegociacao.payment_link || 'Link não disponível';
    const orderId = dadosNegociacao.order_id || 'N/A';
    const isTemp = linkPagamento.includes('temp_') || linkPagamento === 'Aguardando geração...';
    
    content.innerHTML = `
        <div class="card bg-light">
            <div class="card-body">
                <h5 class="card-title">
                    <i class="fas fa-link me-2"></i>Link de Pagamento ${isTemp ? '(Temporário)' : '(Pagar.me)'}
                </h5>
                
                <div class="alert alert-${isTemp ? 'warning' : 'success'} mb-3">
                    <div class="row">
                        <div class="col-md-4">
                            <small><strong>Valor a Pagar:</strong></small><br>
                            <h4 class="mb-0">R$ ${dadosNegociacao.valor_total.toFixed(2).replace('.', ',')}</h4>
                        </div>
                        <div class="col-md-4">
                            <small><strong>Negociação ID:</strong></small><br>
                            <h4 class="mb-0">#${dadosNegociacao.negociacao_id}</h4>
                        </div>
                        <div class="col-md-4">
                            <small><strong>Order ID:</strong></small><br>
                            <h6 class="mb-0"><code>${orderId}</code></h6>
                        </div>
                    </div>
                </div>
                
                <div class="mb-3">
                    <label class="form-label"><strong>Link de Pagamento:</strong></label>
                    <div class="input-group">
                        <input type="text" 
                               class="form-control font-monospace" 
                               id="linkPagamentoInput" 
                               value="${linkPagamento}" 
                               readonly>
                        <button class="btn btn-outline-secondary" type="button" onclick="copiarLinkPagamento()">
                            <i class="fas fa-copy"></i> Copiar
                        </button>
                        ${linkPagamento !== 'Aguardando geração...' && linkPagamento !== 'Link não disponível' ? 
                            `<a href="${linkPagamento}" target="_blank" class="btn btn-outline-primary">
                                <i class="fas fa-external-link-alt"></i> Abrir
                            </a>` : ''}
                    </div>
                </div>
                
                ${isTemp ? `
                    <div class="alert alert-warning">
                        <i class="fas fa-info-circle me-2"></i>
                        <strong>Modo de Testes:</strong> Este é um link temporário. Configure a Secret Key do Pagar.me em <code>includes/pagarme_config.php</code> para gerar links reais.
                    </div>
                ` : `
                    <div class="alert alert-success">
                        <i class="fas fa-check-circle me-2"></i>
                        <strong>Link do Pagar.me gerado!</strong> Este é um link oficial de pagamento. O cliente pode pagar via PIX ou Cartão de Crédito.
                    </div>
                `}
                
                <div class="alert alert-info">
                    <i class="fas fa-lightbulb me-2"></i>
                    <strong>Dica:</strong> Você pode acompanhar todos os links gerados na página <a href="?page=logs_pagamento">Logs Pagar.me</a>
                </div>
            </div>
        </div>
    `;
    
    modal.style.display = 'flex';
}

// Gerar link de pagamento temporário (será substituído pela API Pagar.me)
function gerarLinkPagamentoTemporario(dados) {
    // Por enquanto, gerar um link placeholder
    // Futuramente será substituído pelo link real do Pagar.me
    const baseUrl = window.location.origin + '/portal';
    return `${baseUrl}/pagamento.php?negociacao_id=${dados.negociacao_id}&aluno_id=${dados.aluno_id}&valor=${dados.valor_total.toFixed(2)}`;
}

// Copiar link de pagamento
function copiarLinkPagamento() {
    const linkInput = document.getElementById('linkPagamentoInput');
    if (linkInput) {
        linkInput.select();
        document.execCommand('copy');
        
        showSuccessMessage('✅ Link copiado para a área de transferência!');
        
        // Feedback visual
        const btn = event.target.closest('button');
        const originalHTML = btn.innerHTML;
        btn.innerHTML = '<i class="fas fa-check"></i> Copiado!';
        btn.classList.add('btn-success');
        
        setTimeout(() => {
            btn.innerHTML = originalHTML;
            btn.classList.remove('btn-success');
        }, 2000);
    }
}

// Enviar via WhatsApp
function enviarViaWhatsApp() {
    if (!dadosPagamentoAtual) {
        alert('Erro: dados de pagamento não disponíveis');
        return;
    }
    
    const linkInput = document.getElementById('linkPagamentoInput');
    const linkPagamento = linkInput ? linkInput.value : '';
    
    // Buscar dados do lead para obter o telefone
    const leadId = dadosPagamentoAtual.lead_id;
    const lead = NEGOCIACOES_DATA.find(item => item.id == leadId);
    
    if (!lead || !lead.telefone) {
        alert('❌ Telefone do lead não encontrado');
        return;
    }
    
    // Remover formatação do telefone
    const telefoneNumeros = lead.telefone.replace(/\D/g, '');
    
    // Montar mensagem
    const mensagem = `Olá ${lead.nome}! 😊\n\n` +
                    `Sua negociação foi criada com sucesso! ✅\n\n` +
                    `💰 Valor: R$ ${dadosPagamentoAtual.valor_total.toFixed(2).replace('.', ',')}\n\n` +
                    `Para realizar o pagamento, acesse o link abaixo:\n` +
                    `${linkPagamento}\n\n` +
                    `Qualquer dúvida, estou à disposição!`;
    
    const whatsappUrl = `https://wa.me/55${telefoneNumeros}?text=${encodeURIComponent(mensagem)}`;
    
    window.open(whatsappUrl, '_blank');
}

// Fechar modal de link de pagamento
function closeLinkPagamentoModal() {
    const modal = document.getElementById('linkPagamentoModal');
    modal.style.display = 'none';
    dadosPagamentoAtual = null;
}

// ===== FIM DAS FUNÇÕES DO MODAL DE NEGOCIAÇÃO =====

// Funções do modal de detalhes
async function openDetailsModal(itemId) {
    // Buscar os dados da negociação/lead pelo ID
    const item = NEGOCIACOES_DATA.find(neg => neg.id == itemId);
    
    if (!item) {
        alert('Item não encontrado');
        return;
    }
    
    const modal = document.getElementById('detailsModal');
    const title = document.getElementById('modalTitle');
    
    // Preencher título
    title.textContent = item.tipo === 'lead' ? 'Detalhes do Lead' : 'Detalhes da Negociação';
    
    // Preencher campos
    document.getElementById('itemId').value = item.id;
    document.getElementById('itemTipo').value = item.tipo || 'negociacao';
    document.getElementById('editNome').value = item.nome || '';
    document.getElementById('editEmail').value = item.email || '';
    document.getElementById('editTelefone').value = item.telefone || '';
    document.getElementById('editProfissao').value = item.profissao || '';
    document.getElementById('editValor').value = item.valor || '';
    document.getElementById('editStatus').value = item.status || 'aguardando';
    document.getElementById('editObservacoes').value = item.observacoes || '';
    
    // Preencher horário de preferência (readonly - vem do lead)
    const horarioPreferenciaInput = document.getElementById('editHorarioPreferencia');
    if (horarioPreferenciaInput) {
        horarioPreferenciaInput.value = item.horario_preferencia || 'Não informado';
    }
    
    // Preencher área de interesse (readonly - vem do lead)
    const areaInteresseInput = document.getElementById('editAreaInteresse');
    if (areaInteresseInput) {
        areaInteresseInput.value = item.area_interesse || 'Não informado';
    }
    
    // Preencher data e horário de agendamento se disponíveis (editável)
    const dataAgendamentoInput = document.getElementById('editDataAgendamento');
    const horarioAgendamentoInput = document.getElementById('editHorarioAgendamento');
    
    if (dataAgendamentoInput) {
        dataAgendamentoInput.value = item.data_agendamento || '';
    }
    if (horarioAgendamentoInput) {
        horarioAgendamentoInput.value = item.horario_agendamento || '';
    }
    
    // Mostrar/ocultar campo valor baseado no tipo
    const valorGroup = document.getElementById('valorGroup');
    if (item.tipo === 'lead') {
        valorGroup.style.display = 'none';
    } else {
        valorGroup.style.display = 'block';
    }
    
    // Mostrar modal
    modal.style.display = 'block';
    document.body.style.overflow = 'hidden';
}

function closeDetailsModal() {
    const modal = document.getElementById('detailsModal');
    modal.style.display = 'none';
    document.body.style.overflow = 'auto';
}

function saveDetails() {
    const form = document.getElementById('detailsForm');
    const formData = new FormData(form);
    
    // Converter FormData para objeto
    const data = {};
    formData.forEach((value, key) => {
        data[key] = value;
    });
    
    console.log('Enviando dados:', data);
    
    // Desabilitar botão durante o envio
    const saveBtn = document.querySelector('.btn-save');
    const originalText = saveBtn.textContent;
    saveBtn.textContent = 'Salvando...';
    saveBtn.disabled = true;
    
    // Fazer requisição AJAX para salvar
    fetch('../ajax/update_details.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify(data)
    })
    .then(response => {
        console.log('Response status:', response.status);
        return response.text().then(text => {
            console.log('Response text:', text);
            try {
                return JSON.parse(text);
            } catch (e) {
                throw new Error('Resposta inválida do servidor: ' + text);
            }
        });
    })
    .then(result => {
        console.log('Result:', result);
        if (result.success) {
            closeDetailsModal();
            // Recarregar o kanban
            renderKanbanBoard();
            showSuccessMessage('Dados atualizados com sucesso!');
        } else {
            alert('Erro ao salvar: ' + (result.message || 'Erro desconhecido'));
        }
    })
    .catch(error => {
        console.error('Erro:', error);
        alert('Erro na comunicação com o servidor: ' + error.message);
    })
    .finally(() => {
        // Reabilitar botão
        saveBtn.textContent = originalText;
        saveBtn.disabled = false;
    });
}

function showSuccessMessage(message) {
    const alert = document.createElement('div');
    alert.className = 'alert alert-success alert-dismissible fade show';
    alert.innerHTML = `
        <i class="fas fa-check-circle me-2"></i>
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.querySelector('.custom-card .card-body').insertBefore(alert, document.querySelector('.kanban-container'));
    
    setTimeout(() => {
        alert.remove();
    }, 3000);
}

// Fechar modal clicando fora dele
document.addEventListener('click', function(e) {
    const modal = document.getElementById('detailsModal');
    if (e.target === modal) {
        closeDetailsModal();
    }
});

// Fechar modal com ESC
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeDetailsModal();
        closeLossModal();
    }
});

// Inicializar quando a página carregar
document.addEventListener('DOMContentLoaded', function() {
    renderKanbanBoard();
    // Forçar atualização dos contadores e limpeza de estados inconsistentes
    setTimeout(() => {
        updateColumnCounts();
    }, 100);
    
    // Iniciar verificação automática de mudanças de status
    iniciarVerificacaoAutomatica();
});

// Verificação automática de mudanças de status (polling)
let ultimaVerificacao = new Date().toISOString();
let intervalVerificacao = null;

function iniciarVerificacaoAutomatica() {
    // Verificar a cada 30 segundos se há mudanças
    intervalVerificacao = setInterval(async () => {
        await verificarMudancasStatus();
    }, 30000); // 30 segundos
    
    // Verificar também quando a página recebe foco
    document.addEventListener('visibilitychange', function() {
        if (!document.hidden) {
            verificarMudancasStatus();
        }
    });
}

async function verificarMudancasStatus() {
    try {
        const response = await fetch(`/portal/especialista/ajax/verificar_status_kanban.php?last_check=${encodeURIComponent(ultimaVerificacao)}`);
        const data = await response.json();
        
        if (!data.success) {
            console.error('Erro ao verificar mudanças:', data.message);
            return;
        }
        
        // Se houver aprovações automáticas, recarregar o Kanban
        if (data.aprovacoes_automaticas && data.aprovacoes_automaticas.length > 0) {
            console.log('🎉 Aprovações automáticas detectadas:', data.aprovacoes_automaticas);
            showSuccessMessage(`✅ ${data.aprovacoes_automaticas.length} negociação(ões) aprovada(s) automaticamente!`);
            
            // Recarregar dados do Kanban
            location.reload();
            return;
        }
        
        // Se houver mudanças de status, atualizar cards específicos
        if (data.mudancas_negociacoes && data.mudancas_negociacoes.length > 0) {
            console.log('📊 Mudanças detectadas:', data.mudancas_negociacoes);
            
            // Atualizar cards afetados
            data.mudancas_negociacoes.forEach(mudanca => {
                const cardId = `neg-${mudanca.id}`;
                const card = document.querySelector(`[data-id="${cardId}"]`);
                
                if (card && card.dataset.status !== mudanca.status_kanban) {
                    // Card mudou de status - mover para coluna correta
                    const targetColumn = document.querySelector(`[data-status="${mudanca.status_kanban}"] .kanban-cards`);
                    if (targetColumn) {
                        targetColumn.appendChild(card);
                        card.dataset.status = mudanca.status_kanban;
                        updateColumnCounts();
                        
                        // Se foi aprovado, mostrar mensagem
                        if (mudanca.status_kanban === 'aprovada') {
                            showSuccessMessage(`✅ Negociação #${mudanca.id} aprovada automaticamente!`);
                        }
                    }
                }
            });
        }
        
        if (data.mudancas_leads && data.mudancas_leads.length > 0) {
            console.log('📊 Mudanças em leads detectadas:', data.mudancas_leads);
            
            data.mudancas_leads.forEach(mudanca => {
                const cardId = `lead-${mudanca.id}`;
                const card = document.querySelector(`[data-id="${cardId}"]`);
                
                if (card && card.dataset.status !== mudanca.status_kanban) {
                    const targetColumn = document.querySelector(`[data-status="${mudanca.status_kanban}"] .kanban-cards`);
                    if (targetColumn) {
                        targetColumn.appendChild(card);
                        card.dataset.status = mudanca.status_kanban;
                        updateColumnCounts();
                    }
                }
            });
        }
        
        // Atualizar timestamp da última verificação
        ultimaVerificacao = data.timestamp || new Date().toISOString();
        
    } catch (error) {
        console.error('Erro ao verificar mudanças de status:', error);
    }
}
</script>

