<?php
// A página é incluída pelo index.php, então $pdo já está disponível
if (!isset($_SESSION['especialista_id'])) {
    echo '<div class="alert alert-danger">Sessão inválida. Faça login novamente.</div>';
    exit();
}

// Garantir que $pdo existe
if (!isset($pdo)) {
    require_once '../../includes/config.php';
}

$especialista_id = $_SESSION['especialista_id'];

// Filtros
$filtro_status = $_GET['status'] ?? 'todas';
$filtro_periodo = $_GET['periodo'] ?? 'mes';

// Definir período
switch ($filtro_periodo) {
    case 'semana':
        $data_inicio = date('Y-m-d', strtotime('monday this week'));
        break;
    case 'mes':
    default:
        $data_inicio = date('Y-m-01');
        break;
    case 'trimestre':
        $mes_atual = date('n');
        $trimestre_inicio = floor(($mes_atual - 1) / 3) * 3 + 1;
        $data_inicio = date('Y-' . str_pad($trimestre_inicio, 2, '0', STR_PAD_LEFT) . '-01');
        break;
    case 'ano':
        $data_inicio = date('Y-01-01');
        break;
}

// Buscar comissões
$where_clauses = ["c.especialista_id = ?", "c.data_criacao >= ?"];
$params = [$especialista_id, $data_inicio];

if ($filtro_status !== 'todas') {
    $where_clauses[] = "c.status = ?";
    $params[] = $filtro_status;
}

$where_sql = implode(' AND ', $where_clauses);

$stmt = $pdo->prepare("
    SELECT 
        c.*,
        n.valor_total as valor_venda,
        n.forma_pagamento,
        n.numero_parcelas,
        a.nome as aluno_nome,
        a.email as aluno_email,
        l.nome as lead_nome
    FROM comissoes c
    INNER JOIN negociacoes n ON c.negociacao_id = n.id
    INNER JOIN alunos a ON n.aluno_id = a.id
    LEFT JOIN vinculo v ON n.aluno_id = v.aluno_id AND v.status = 'ativo'
    LEFT JOIN leads l ON v.lead_id = l.id
    WHERE {$where_sql}
    ORDER BY c.data_criacao DESC
");
$stmt->execute($params);
$comissoes = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Estatísticas
$stats = [
    'total_comissoes' => 0,
    'valor_pendente' => 0,
    'valor_pago' => 0,
    'valor_total' => 0,
    'qtd_pendente' => 0,
    'qtd_paga' => 0
];

foreach ($comissoes as $com) {
    $stats['total_comissoes']++;
    $stats['valor_total'] += $com['valor'];
    
    if ($com['status'] == 'pendente') {
        $stats['qtd_pendente']++;
        $stats['valor_pendente'] += $com['valor'];
    } elseif ($com['status'] == 'paga') {
        $stats['qtd_paga']++;
        $stats['valor_pago'] += $com['valor'];
    }
}

// Comissões por mês (últimos 6 meses)
$stmt = $pdo->prepare("
    SELECT 
        DATE_FORMAT(c.data_criacao, '%Y-%m') as mes,
        COUNT(*) as quantidade,
        SUM(c.valor) as valor_total,
        SUM(CASE WHEN c.status = 'paga' THEN c.valor ELSE 0 END) as valor_pago
    FROM comissoes c
    WHERE c.especialista_id = ?
    AND c.data_criacao >= DATE_SUB(NOW(), INTERVAL 6 MONTH)
    GROUP BY DATE_FORMAT(c.data_criacao, '%Y-%m')
    ORDER BY mes
");
$stmt->execute([$especialista_id]);
$comissoes_por_mes = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<div class="page-header">
    <h1 class="page-title">
        <i class="fas fa-percentage me-2"></i>
        Minhas Comissões
    </h1>
    <p class="page-subtitle">Acompanhe seus ganhos e comissões de vendas</p>
</div>

<!-- Estatísticas -->
<div class="row mb-4">
    <div class="col-md-3">
        <div class="card text-white bg-primary">
            <div class="card-body text-center">
                <i class="fas fa-list-alt fa-2x mb-3"></i>
                <h2 class="mb-0"><?php echo $stats['total_comissoes']; ?></h2>
                <p class="mb-0">Total de Comissões</p>
                <small class="opacity-75">No período</small>
            </div>
        </div>
    </div>
    
    <div class="col-md-3">
        <div class="card text-white bg-warning">
            <div class="card-body text-center">
                <i class="fas fa-clock fa-2x mb-3"></i>
                <h2 class="mb-0">R$ <?php echo number_format($stats['valor_pendente'], 0, ',', '.'); ?></h2>
                <p class="mb-0">A Receber</p>
                <small class="opacity-75"><?php echo $stats['qtd_pendente']; ?> comissões</small>
            </div>
        </div>
    </div>
    
    <div class="col-md-3">
        <div class="card text-white bg-success">
            <div class="card-body text-center">
                <i class="fas fa-check-circle fa-2x mb-3"></i>
                <h2 class="mb-0">R$ <?php echo number_format($stats['valor_pago'], 0, ',', '.'); ?></h2>
                <p class="mb-0">Recebido</p>
                <small class="opacity-75"><?php echo $stats['qtd_paga']; ?> comissões</small>
            </div>
        </div>
    </div>
    
    <div class="col-md-3">
        <div class="card text-white bg-info">
            <div class="card-body text-center">
                <i class="fas fa-dollar-sign fa-2x mb-3"></i>
                <h2 class="mb-0">R$ <?php echo number_format($stats['valor_total'], 0, ',', '.'); ?></h2>
                <p class="mb-0">Total Ganho</p>
                <small class="opacity-75">No período</small>
            </div>
        </div>
    </div>
</div>

<!-- Gráfico de Evolução -->
<div class="card mb-4">
    <div class="card-header">
        <h5 class="card-title mb-0">
            <i class="fas fa-chart-line me-2"></i>
            Evolução de Comissões (Últimos 6 Meses)
        </h5>
    </div>
    <div class="card-body">
        <canvas id="graficoComissoes" height="80"></canvas>
    </div>
</div>

<!-- Filtros -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" class="row g-3">
            <input type="hidden" name="page" value="comissoes">
            
            <div class="col-md-4">
                <label class="form-label">Status</label>
                <select name="status" class="form-select" onchange="this.form.submit()">
                    <option value="todas" <?php echo $filtro_status == 'todas' ? 'selected' : ''; ?>>Todas</option>
                    <option value="pendente" <?php echo $filtro_status == 'pendente' ? 'selected' : ''; ?>>Pendentes</option>
                    <option value="paga" <?php echo $filtro_status == 'paga' ? 'selected' : ''; ?>>Pagas</option>
                </select>
            </div>
            
            <div class="col-md-4">
                <label class="form-label">Período</label>
                <select name="periodo" class="form-select" onchange="this.form.submit()">
                    <option value="semana" <?php echo $filtro_periodo == 'semana' ? 'selected' : ''; ?>>Esta Semana</option>
                    <option value="mes" <?php echo $filtro_periodo == 'mes' ? 'selected' : ''; ?>>Este Mês</option>
                    <option value="trimestre" <?php echo $filtro_periodo == 'trimestre' ? 'selected' : ''; ?>>Este Trimestre</option>
                    <option value="ano" <?php echo $filtro_periodo == 'ano' ? 'selected' : ''; ?>>Este Ano</option>
                </select>
            </div>
            
            <div class="col-md-4 d-flex align-items-end">
                <button type="button" class="btn btn-secondary w-100" onclick="window.location.href='?page=comissoes'">
                    <i class="fas fa-redo me-2"></i>Limpar Filtros
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Lista de Comissões -->
<div class="card">
    <div class="card-header">
        <h5 class="card-title mb-0">
            <i class="fas fa-list me-2"></i>
            Minhas Comissões (<?php echo count($comissoes); ?>)
        </h5>
    </div>
    <div class="card-body">
        <?php if (empty($comissoes)): ?>
            <div class="text-center py-5 text-muted">
                <i class="fas fa-inbox fa-3x mb-3"></i>
                <p>Nenhuma comissão encontrada no período selecionado</p>
            </div>
        <?php else: ?>
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Cliente</th>
                            <th>Lead Origem</th>
                            <th>Valor da Venda</th>
                            <th>Comissão (10%)</th>
                            <th>Status</th>
                            <th>Data Geração</th>
                            <th>Data Pagamento</th>
                            <th>Ações</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($comissoes as $com): ?>
                            <tr>
                                <td><strong>#<?php echo $com['id']; ?></strong></td>
                                <td>
                                    <div>
                                        <strong><?php echo htmlspecialchars($com['aluno_nome']); ?></strong>
                                        <br>
                                        <small class="text-muted"><?php echo htmlspecialchars($com['aluno_email']); ?></small>
                                    </div>
                                </td>
                                <td>
                                    <?php if ($com['lead_nome']): ?>
                                        <span class="badge bg-info">
                                            <i class="fas fa-user-tag"></i> <?php echo htmlspecialchars($com['lead_nome']); ?>
                                        </span>
                                    <?php else: ?>
                                        <span class="text-muted">-</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <strong class="text-primary">R$ <?php echo number_format($com['valor_venda'], 2, ',', '.'); ?></strong>
                                    <br>
                                    <small class="text-muted">
                                        <?php echo $com['numero_parcelas']; ?>x - <?php echo $com['forma_pagamento']; ?>
                                    </small>
                                </td>
                                <td>
                                    <strong class="text-success fs-5">R$ <?php echo number_format($com['valor'], 2, ',', '.'); ?></strong>
                                </td>
                                <td>
                                    <?php if ($com['status'] == 'pendente'): ?>
                                        <span class="badge bg-warning">
                                            <i class="fas fa-clock"></i> Pendente
                                        </span>
                                    <?php elseif ($com['status'] == 'paga'): ?>
                                        <span class="badge bg-success">
                                            <i class="fas fa-check-circle"></i> Paga
                                        </span>
                                    <?php else: ?>
                                        <span class="badge bg-secondary">
                                            <?php echo ucfirst($com['status']); ?>
                                        </span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <small class="text-muted">
                                        <?php echo date('d/m/Y', strtotime($com['data_criacao'])); ?>
                                    </small>
                                </td>
                                <td>
                                    <?php if ($com['data_pagamento']): ?>
                                        <small class="text-success">
                                            <i class="fas fa-calendar-check"></i>
                                            <?php echo date('d/m/Y', strtotime($com['data_pagamento'])); ?>
                                        </small>
                                    <?php else: ?>
                                        <span class="text-muted">-</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <button class="btn btn-sm btn-info" onclick="verDetalhesVenda(<?php echo $com['negociacao_id']; ?>)">
                                        <i class="fas fa-eye"></i> Ver Venda
                                    </button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                    <tfoot>
                        <tr class="table-secondary">
                            <td colspan="4" class="text-end"><strong>TOTAL:</strong></td>
                            <td><strong class="text-success fs-5">R$ <?php echo number_format($stats['valor_total'], 2, ',', '.'); ?></strong></td>
                            <td colspan="4"></td>
                        </tr>
                    </tfoot>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Modal Detalhes da Venda -->
<div class="modal fade" id="modalDetalhesVenda" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="fas fa-info-circle me-2"></i>
                    Detalhes da Venda
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="detalhes-venda-conteudo">
                <div class="text-center py-4">
                    <i class="fas fa-spinner fa-spin fa-2x"></i>
                    <p class="mt-2">Carregando...</p>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="alert alert-info mt-4">
    <h6><i class="fas fa-info-circle me-2"></i>Como Funcionam as Comissões:</h6>
    <ul class="mb-0">
        <li>Você recebe <strong>10% do valor total</strong> de cada venda aprovada</li>
        <li>A comissão é gerada automaticamente quando a venda é aprovada pelo admin</li>
        <li>Status <strong>Pendente</strong>: Aguardando processamento do pagamento</li>
        <li>Status <strong>Paga</strong>: Comissão já foi creditada para você</li>
        <li>O pagamento geralmente ocorre até o 5º dia útil do mês seguinte</li>
    </ul>
</div>

<script>
// Dados para o gráfico
const comissoesPorMes = <?php echo json_encode($comissoes_por_mes); ?>;

// Gráfico de Comissões
const ctx = document.getElementById('graficoComissoes').getContext('2d');
new Chart(ctx, {
    type: 'bar',
    data: {
        labels: comissoesPorMes.map(c => {
            const [ano, mes] = c.mes.split('-');
            const meses = ['Jan', 'Fev', 'Mar', 'Abr', 'Mai', 'Jun', 'Jul', 'Ago', 'Set', 'Out', 'Nov', 'Dez'];
            return meses[parseInt(mes) - 1] + '/' + ano.substr(2);
        }),
        datasets: [{
            label: 'Comissões Geradas (R$)',
            data: comissoesPorMes.map(c => c.valor_total),
            backgroundColor: 'rgba(54, 162, 235, 0.6)',
            borderColor: 'rgb(54, 162, 235)',
            borderWidth: 2
        }, {
            label: 'Comissões Pagas (R$)',
            data: comissoesPorMes.map(c => c.valor_pago),
            backgroundColor: 'rgba(75, 192, 192, 0.6)',
            borderColor: 'rgb(75, 192, 192)',
            borderWidth: 2
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: true,
        interaction: {
            mode: 'index',
            intersect: false
        },
        plugins: {
            legend: {
                display: true,
                position: 'top'
            },
            tooltip: {
                callbacks: {
                    label: function(context) {
                        let label = context.dataset.label || '';
                        if (label) {
                            label += ': ';
                        }
                        if (context.parsed.y !== null) {
                            label += 'R$ ' + context.parsed.y.toFixed(2).replace('.', ',');
                        }
                        return label;
                    }
                }
            }
        },
        scales: {
            y: {
                beginAtZero: true,
                title: {
                    display: true,
                    text: 'Valor (R$)'
                },
                ticks: {
                    callback: function(value) {
                        return 'R$ ' + value.toFixed(0);
                    }
                }
            }
        }
    }
});

function verDetalhesVenda(negociacaoId) {
    const modal = new bootstrap.Modal(document.getElementById('modalDetalhesVenda'));
    modal.show();
    
    // Buscar detalhes via AJAX
    fetch(`ajax/get_negociacao_detalhes.php?id=${negociacaoId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                document.getElementById('detalhes-venda-conteudo').innerHTML = data.html;
            } else {
                document.getElementById('detalhes-venda-conteudo').innerHTML = `
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        ${data.message || 'Erro ao carregar detalhes'}
                    </div>
                `;
            }
        })
        .catch(error => {
            document.getElementById('detalhes-venda-conteudo').innerHTML = `
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    Erro ao carregar detalhes
                </div>
            `;
        });
}
</script>

