<?php
require_once '../includes/config.php';

$especialista_id = $_SESSION['especialista_id'];

// Filtros
$filtro_aceite = $_GET['aceite'] ?? 'todos';
$filtro_negociacao = $_GET['negociacao'] ?? 'todas';
$filtro_periodo = $_GET['periodo'] ?? 'todos';

// Definir período para filtro de data
$where_data = "";
switch ($filtro_periodo) {
    case 'semana':
        $where_data = " AND a.data_cadastro >= DATE_SUB(NOW(), INTERVAL 7 DAY)";
        break;
    case 'mes':
        $where_data = " AND a.data_cadastro >= DATE_SUB(NOW(), INTERVAL 30 DAY)";
        break;
    case 'ano':
        $where_data = " AND a.data_cadastro >= DATE_SUB(NOW(), INTERVAL 1 YEAR)";
        break;
    case 'todos':
    default:
        $where_data = "";
        break;
}

// Buscar alunos do especialista com informações de aceite e negociações
$stmt = $pdo->prepare("
    SELECT 
        a.id,
        a.nome,
        a.email,
        a.cpf,
        a.celular,
        a.data_cadastro,
        a.data_aceite_contrato,
        a.primeiro_acesso,
        a.status,
        a.tipo_contrato,
        CASE 
            WHEN a.data_aceite_contrato IS NOT NULL THEN 'realizado'
            WHEN a.primeiro_acesso = 1 AND a.data_aceite_contrato IS NULL THEN 'pendente'
            ELSE 'não_aplicavel'
        END as status_aceite,
        -- Dados de negociação (mais recente)
        (SELECT n.id FROM negociacoes n WHERE n.aluno_id = a.id AND n.especialista_id = ? ORDER BY n.data_criacao DESC LIMIT 1) as negociacao_id,
        (SELECT n.valor_total FROM negociacoes n WHERE n.aluno_id = a.id AND n.especialista_id = ? ORDER BY n.data_criacao DESC LIMIT 1) as valor_total,
        (SELECT n.num_parcelas FROM negociacoes n WHERE n.aluno_id = a.id AND n.especialista_id = ? ORDER BY n.data_criacao DESC LIMIT 1) as num_parcelas,
        (SELECT n.investimento_parcial FROM negociacoes n WHERE n.aluno_id = a.id AND n.especialista_id = ? ORDER BY n.data_criacao DESC LIMIT 1) as investimento_parcial,
        (SELECT n.valor_pago FROM negociacoes n WHERE n.aluno_id = a.id AND n.especialista_id = ? ORDER BY n.data_criacao DESC LIMIT 1) as valor_pago,
        (SELECT n.saldo_restante FROM negociacoes n WHERE n.aluno_id = a.id AND n.especialista_id = ? ORDER BY n.data_criacao DESC LIMIT 1) as saldo_restante,
        (SELECT n.status FROM negociacoes n WHERE n.aluno_id = a.id AND n.especialista_id = ? ORDER BY n.data_criacao DESC LIMIT 1) as negociacao_status,
        (SELECT n.data_criacao FROM negociacoes n WHERE n.aluno_id = a.id AND n.especialista_id = ? ORDER BY n.data_criacao DESC LIMIT 1) as negociacao_data,
        -- Dados do lead origem
        (SELECT l.nome FROM leads l 
         INNER JOIN vinculo vin ON vin.lead_id = l.id 
         WHERE vin.aluno_id = a.id AND vin.status = 'ativo' 
         LIMIT 1) as lead_nome,
        (SELECT l.id FROM leads l 
         INNER JOIN vinculo vin ON vin.lead_id = l.id 
         WHERE vin.aluno_id = a.id AND vin.status = 'ativo' 
         LIMIT 1) as lead_id
    FROM alunos a
    WHERE a.especialista_id = ? {$where_data}
    ORDER BY a.data_cadastro DESC
");
$stmt->execute([
    $especialista_id, $especialista_id, $especialista_id, $especialista_id, 
    $especialista_id, $especialista_id, $especialista_id, $especialista_id,
    $especialista_id
]);
$alunos_raw = $stmt->fetchAll();

// Aplicar filtros de status
$alunos = array_filter($alunos_raw, function($aluno) use ($filtro_aceite, $filtro_negociacao) {
    // Filtro de aceite
    if ($filtro_aceite !== 'todos') {
        if ($aluno['status_aceite'] !== $filtro_aceite) {
            return false;
        }
    }
    
    // Filtro de negociação
    if ($filtro_negociacao !== 'todas') {
        if (empty($aluno['negociacao_status'])) {
            return false;
        }
        if ($aluno['negociacao_status'] !== $filtro_negociacao) {
            return false;
        }
    }
    
    return true;
});

// Contar estatísticas (usar alunos filtrados)
$total_alunos = count($alunos);
$aceites_realizados = count(array_filter($alunos, function($aluno) {
    return $aluno['status_aceite'] === 'realizado';
}));
$aceites_pendentes = count(array_filter($alunos, function($aluno) {
    return $aluno['status_aceite'] === 'pendente';
}));

// Estatísticas de negociações
$total_negociacoes = count(array_filter($alunos, function($aluno) {
    return !empty($aluno['negociacao_id']);
}));
$negociacoes_aprovadas = count(array_filter($alunos, function($aluno) {
    return $aluno['negociacao_status'] === 'aprovada';
}));
$negociacoes_pendentes = count(array_filter($alunos, function($aluno) {
    return $aluno['negociacao_status'] === 'pendente';
}));
$valor_total_negociado = array_sum(array_map(function($aluno) {
    return (float)($aluno['valor_total'] ?? 0);
}, $alunos));

// Estatísticas gerais (sem filtro) para referência
$total_alunos_geral = count($alunos_raw);
$total_negociacoes_geral = count(array_filter($alunos_raw, function($aluno) {
    return !empty($aluno['negociacao_id']);
}));
?>

<div class="page-header mb-4">
    <h1 class="page-title">
        <i class="fas fa-users me-2"></i>
        Meus Alunos e Negociações
    </h1>
    <p class="page-subtitle">
        Gerencie seus alunos e acompanhe todas as negociações em um só lugar
        <br>
        <small class="text-muted">
            <i class="fas fa-info-circle"></i> 
            Total geral: <?php echo $total_alunos_geral; ?> aluno(s) | 
            <?php echo $total_negociacoes_geral; ?> negociação(ões)
            <?php if ($filtro_aceite !== 'todos' || $filtro_negociacao !== 'todas' || $filtro_periodo !== 'todos'): ?>
                | Exibindo: <?php echo count($alunos); ?> (filtrado)
            <?php endif; ?>
        </small>
    </p>
</div>

<div class="row mb-4">
    <div class="col-md-3">
        <div class="custom-card text-center">
            <div class="card-body">
                <i class="fas fa-users fa-2x mb-2 text-primary"></i>
                <h4><?php echo $total_alunos; ?></h4>
                <p class="mb-0 text-secondary">Total de Alunos</p>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="custom-card text-center">
            <div class="card-body">
                <i class="fas fa-handshake fa-2x mb-2 text-info"></i>
                <h4><?php echo $total_negociacoes; ?></h4>
                <p class="mb-0 text-secondary">Negociações</p>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="custom-card text-center">
            <div class="card-body">
                <i class="fas fa-check-circle fa-2x mb-2 text-success"></i>
                <h4><?php echo $negociacoes_aprovadas; ?></h4>
                <p class="mb-0 text-secondary">Aprovadas</p>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="custom-card text-center">
            <div class="card-body">
                <i class="fas fa-dollar-sign fa-2x mb-2 text-success"></i>
                <h4>R$ <?php echo number_format($valor_total_negociado, 0, ',', '.'); ?></h4>
                <p class="mb-0 text-secondary">Valor Total</p>
            </div>
        </div>
    </div>
</div>

<!-- Filtros -->
<div class="custom-card mb-4">
    <div class="card-body">
        <form method="GET" class="row g-3">
            <input type="hidden" name="page" value="meus_alunos">
            
            <div class="col-md-3">
                <label class="form-label small"><i class="fas fa-filter me-1"></i>Status de Aceite</label>
                <select name="aceite" class="form-select form-select-sm" onchange="this.form.submit()">
                    <option value="todos" <?php echo $filtro_aceite == 'todos' ? 'selected' : ''; ?>>Todos</option>
                    <option value="realizado" <?php echo $filtro_aceite == 'realizado' ? 'selected' : ''; ?>>Contratos Aceitos</option>
                    <option value="pendente" <?php echo $filtro_aceite == 'pendente' ? 'selected' : ''; ?>>Aceites Pendentes</option>
                </select>
            </div>
            
            <div class="col-md-3">
                <label class="form-label small"><i class="fas fa-handshake me-1"></i>Status de Negociação</label>
                <select name="negociacao" class="form-select form-select-sm" onchange="this.form.submit()">
                    <option value="todas" <?php echo $filtro_negociacao == 'todas' ? 'selected' : ''; ?>>Todas</option>
                    <option value="pendente" <?php echo $filtro_negociacao == 'pendente' ? 'selected' : ''; ?>>Pendentes</option>
                    <option value="aprovada" <?php echo $filtro_negociacao == 'aprovada' ? 'selected' : ''; ?>>Aprovadas</option>
                    <option value="rejeitada" <?php echo $filtro_negociacao == 'rejeitada' ? 'selected' : ''; ?>>Rejeitadas</option>
                    <option value="cancelada" <?php echo $filtro_negociacao == 'cancelada' ? 'selected' : ''; ?>>Canceladas</option>
                </select>
            </div>
            
            <div class="col-md-3">
                <label class="form-label small"><i class="fas fa-calendar me-1"></i>Período de Cadastro</label>
                <select name="periodo" class="form-select form-select-sm" onchange="this.form.submit()">
                    <option value="todos" <?php echo $filtro_periodo == 'todos' ? 'selected' : ''; ?>>Todo o Período</option>
                    <option value="semana" <?php echo $filtro_periodo == 'semana' ? 'selected' : ''; ?>>Última Semana</option>
                    <option value="mes" <?php echo $filtro_periodo == 'mes' ? 'selected' : ''; ?>>Último Mês</option>
                    <option value="ano" <?php echo $filtro_periodo == 'ano' ? 'selected' : ''; ?>>Último Ano</option>
                </select>
            </div>
            
            <div class="col-md-3 d-flex align-items-end">
                <button type="button" class="btn btn-secondary btn-sm w-100" onclick="window.location.href='?page=meus_alunos'">
                    <i class="fas fa-redo me-2"></i>Limpar Filtros
                </button>
            </div>
        </form>
    </div>
</div>

<div class="custom-card">
    <div class="card-header d-flex justify-content-between align-items-center flex-wrap gap-2">
        <h5 class="mb-0">
            <i class="fas fa-users me-2"></i>Meus Alunos e Negociações (<?php echo count($alunos); ?>)
        </h5>
        <div class="text-muted small">
            <?php if ($filtro_aceite !== 'todos' || $filtro_negociacao !== 'todas' || $filtro_periodo !== 'todos'): ?>
                <i class="fas fa-filter"></i> Filtros ativos
            <?php endif; ?>
        </div>
    </div>
    <div class="card-body p-0">
        <?php if (empty($alunos)): ?>
            <div class="alert alert-info text-center m-4">
                <i class="fas fa-info-circle me-2"></i>
                <strong>Nenhum aluno cadastrado ainda.</strong><br>
                Compartilhe seu <a href="index.php?page=meu_link">link personalizado</a> para que alunos possam se cadastrar e sejam vinculados a você.
            </div>
        <?php else: ?>
            <div class="alunos-list-container">
                        <?php foreach ($alunos as $aluno): ?>
                    <div class="aluno-card aluno-row" data-status="<?php echo $aluno['status_aceite']; ?>">
                        <div class="aluno-card-header" onclick="toggleCard(this)">
                            <div class="d-flex align-items-center flex-grow-1" style="gap: 1rem;">
                                <div class="aluno-avatar">
                                    <div class="avatar-circle">
                                                <?php echo strtoupper(substr($aluno['nome'], 0, 2)); ?>
                                            </div>
                                        </div>
                                <div class="aluno-info-main flex-grow-1">
                                    <div class="aluno-name">
                                        <strong><?php echo htmlspecialchars($aluno['nome']); ?></strong>
                                            <?php if ($aluno['status'] === 'inativo'): ?>
                                            <span class="badge bg-secondary ms-2">Inativo</span>
                                        <?php endif; ?>
                                        <?php if ($aluno['lead_nome']): ?>
                                            <span class="badge bg-info ms-2">
                                                <i class="fas fa-user-tag"></i> <?php echo htmlspecialchars($aluno['lead_nome']); ?>
                                            </span>
                                        <?php endif; ?>
                                        <?php if ($aluno['negociacao_status']): ?>
                                            <?php
                                            $status_classes = [
                                                'pendente' => 'warning',
                                                'aprovada' => 'success',
                                                'rejeitada' => 'danger',
                                                'cancelada' => 'secondary'
                                            ];
                                            $status_icons = [
                                                'pendente' => 'clock',
                                                'aprovada' => 'check-circle',
                                                'rejeitada' => 'times-circle',
                                                'cancelada' => 'ban'
                                            ];
                                            $status_texts = [
                                                'pendente' => 'Pendente',
                                                'aprovada' => 'Aprovada',
                                                'rejeitada' => 'Rejeitada',
                                                'cancelada' => 'Cancelada'
                                            ];
                                            $class = $status_classes[$aluno['negociacao_status']] ?? 'secondary';
                                            $icon = $status_icons[$aluno['negociacao_status']] ?? 'info-circle';
                                            $text = $status_texts[$aluno['negociacao_status']] ?? $aluno['negociacao_status'];
                                            ?>
                                            <span class="badge bg-<?php echo $class; ?> ms-2">
                                                <i class="fas fa-<?php echo $icon; ?>"></i> <?php echo $text; ?>
                                            </span>
                                            <?php endif; ?>
                                        </div>
                                    <div class="aluno-contact">
                                        <span class="text-muted">
                                            <i class="fas fa-envelope"></i> <?php echo htmlspecialchars($aluno['email']); ?>
                                        </span>
                                    <?php if ($aluno['celular']): ?>
                                            <span class="text-muted ms-3">
                                        <a href="https://wa.me/55<?php echo preg_replace('/[^0-9]/', '', $aluno['celular']); ?>" 
                                                   target="_blank" class="text-success text-decoration-none"
                                                   onclick="event.stopPropagation();">
                                                    <i class="fab fa-whatsapp"></i> <?php echo formatTelefone($aluno['celular']); ?>
                                        </a>
                                            </span>
                                    <?php endif; ?>
                                        <?php if ($aluno['valor_total']): ?>
                                            <span class="text-success ms-3">
                                                <i class="fas fa-dollar-sign"></i> <strong>R$ <?php echo number_format($aluno['valor_total'], 2, ',', '.'); ?></strong>
                                        </span>
                                    <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            <div class="aluno-header-right">
                                <div class="aluno-actions" onclick="event.stopPropagation();">
                                    <?php if ($aluno['investimento_parcial'] && $aluno['saldo_restante'] > 0): ?>
                                        <button type="button" class="btn btn-sm btn-outline-success" 
                                                onclick="gerarNovoPagamento(<?php echo $aluno['id']; ?>, <?php echo $aluno['negociacao_id']; ?>, <?php echo $aluno['saldo_restante']; ?>)"
                                                title="Gerar novo link de pagamento">
                                            <i class="fas fa-plus-circle"></i> Novo Pagamento
                                        </button>
                                    <?php endif; ?>
                                    <button type="button" class="btn btn-sm btn-outline-info" 
                                                onclick="mostrarDetalhes(<?php echo $aluno['id']; ?>)"
                                                title="Ver detalhes">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                </div>
                                <div class="expand-icon">
                                    <i class="fas fa-chevron-down"></i>
                                </div>
                            </div>
                        </div>
                        
                        <div class="aluno-card-body collapse">
                            <div class="row g-3">
                                <!-- Informações Básicas -->
                                <div class="col-md-6 col-lg-3">
                                    <div class="info-item">
                                        <small class="text-muted d-block mb-1">
                                            <i class="fas fa-calendar"></i> Cadastro
                                        </small>
                                        <span><?php echo date('d/m/Y', strtotime($aluno['data_cadastro'])); ?></span>
                                    </div>
                                </div>
                                <div class="col-md-6 col-lg-3">
                                    <div class="info-item">
                                        <small class="text-muted d-block mb-1">
                                            <i class="fas fa-file-contract"></i> Contrato
                                        </small>
                                        <?php if ($aluno['status_aceite'] === 'realizado'): ?>
                                            <span class="badge bg-success">
                                                <i class="fas fa-check"></i> Aceito
                                            </span>
                                        <?php elseif ($aluno['status_aceite'] === 'pendente'): ?>
                                            <span class="badge bg-warning">
                                                <i class="fas fa-clock"></i> Pendente
                                            </span>
                                        <?php else: ?>
                                            <span class="badge bg-secondary">N/A</span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div class="col-md-6 col-lg-3">
                                    <div class="info-item">
                                        <small class="text-muted d-block mb-1">
                                            <i class="fas fa-tag"></i> Tipo
                                        </small>
                                        <?php if ($aluno['tipo_contrato']): ?>
                                            <span class="badge bg-<?php echo $aluno['tipo_contrato'] == 'regular' ? 'primary' : 'success'; ?>">
                                                <?php echo $aluno['tipo_contrato'] == 'regular' ? 'Regular' : 'Bolsista'; ?>
                                            </span>
                                        <?php else: ?>
                                            <span class="text-muted">-</span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div class="col-md-6 col-lg-3">
                                    <div class="info-item">
                                        <small class="text-muted d-block mb-1">
                                            <i class="fas fa-info-circle"></i> Status Negociação
                                        </small>
                                        <?php if ($aluno['negociacao_status']): ?>
                                            <?php
                                            $status_classes = [
                                                'pendente' => 'warning',
                                                'aprovada' => 'success',
                                                'rejeitada' => 'danger',
                                                'cancelada' => 'secondary'
                                            ];
                                            $status_icons = [
                                                'pendente' => 'clock',
                                                'aprovada' => 'check-circle',
                                                'rejeitada' => 'times-circle',
                                                'cancelada' => 'ban'
                                            ];
                                            $status_texts = [
                                                'pendente' => 'Pendente',
                                                'aprovada' => 'Aprovada',
                                                'rejeitada' => 'Rejeitada',
                                                'cancelada' => 'Cancelada'
                                            ];
                                            $class = $status_classes[$aluno['negociacao_status']] ?? 'secondary';
                                            $icon = $status_icons[$aluno['negociacao_status']] ?? 'info-circle';
                                            $text = $status_texts[$aluno['negociacao_status']] ?? $aluno['negociacao_status'];
                                            ?>
                                            <span class="badge bg-<?php echo $class; ?>">
                                                <i class="fas fa-<?php echo $icon; ?>"></i> <?php echo $text; ?>
                                            </span>
                                        <?php else: ?>
                                            <span class="text-muted">Sem negociação</span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                
                                <!-- Informações Financeiras -->
                                <?php if ($aluno['valor_total'] || $aluno['negociacao_id']): ?>
                                    <div class="col-md-6 col-lg-3">
                                        <div class="info-item">
                                            <small class="text-muted d-block mb-1">
                                                <i class="fas fa-dollar-sign"></i> Valor Total
                                            </small>
                                            <strong class="text-success">R$ <?php echo number_format($aluno['valor_total'] ?? 0, 2, ',', '.'); ?></strong>
                                        </div>
                                    </div>
                                    <div class="col-md-6 col-lg-3">
                                        <div class="info-item">
                                            <small class="text-muted d-block mb-1">
                                                <i class="fas fa-hand-holding-usd"></i> Investimento
                                            </small>
                                            <?php if ($aluno['investimento_parcial']): ?>
                                                <span class="badge bg-warning text-dark">Parcial</span>
                                                <?php if ($aluno['valor_pago']): ?>
                                                    <br><small class="text-success">R$ <?php echo number_format($aluno['valor_pago'], 2, ',', '.'); ?></small>
                                                <?php endif; ?>
                                            <?php elseif ($aluno['negociacao_id']): ?>
                                                <span class="badge bg-info">Total</span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <div class="col-md-6 col-lg-3">
                                        <div class="info-item">
                                            <small class="text-muted d-block mb-1">
                                                <i class="fas fa-calendar-alt"></i> Parcelas
                                            </small>
                                            <?php if ($aluno['num_parcelas'] && $aluno['valor_total']): ?>
                                                <?php 
                                                $num_parcelas = $aluno['num_parcelas'];
                                                $valor_parcela = $num_parcelas > 0 ? ($aluno['valor_total'] / $num_parcelas) : $aluno['valor_total'];
                                                ?>
                                                <span><?php echo $num_parcelas; ?>x de R$ <?php echo number_format($valor_parcela, 2, ',', '.'); ?></span>
                                            <?php else: ?>
                                                <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <div class="col-md-6 col-lg-3">
                                        <div class="info-item">
                                            <small class="text-muted d-block mb-1">
                                                <i class="fas fa-wallet"></i> Pagamento
                                            </small>
                                            <?php if ($aluno['investimento_parcial'] && $aluno['saldo_restante'] > 0): ?>
                                                <span class="text-warning">
                                                    <strong>Saldo: R$ <?php echo number_format($aluno['saldo_restante'], 2, ',', '.'); ?></strong>
                                                </span>
                                            <?php elseif ($aluno['negociacao_status'] == 'aprovada' && $aluno['valor_total']): ?>
                                                <?php 
                                                $valor_pago_calc = $aluno['valor_pago'] ?? 0;
                                                $percentual_recebido = $aluno['valor_total'] > 0 ? ($valor_pago_calc / $aluno['valor_total']) * 100 : 0;
                                                ?>
                                                <div class="progress" style="height: 18px; font-size: 11px;">
                                                    <div class="progress-bar bg-success" style="width: <?php echo $percentual_recebido; ?>%">
                                                        <?php echo number_format($percentual_recebido, 0); ?>%
                                                    </div>
                                                </div>
                                            <?php else: ?>
                                                <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                        <?php endif; ?>
                            </div>
                        </div>
                                    </div>
                        <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<style>
.alunos-list-container {
    padding: 1rem;
}

.aluno-card {
    background: #fff;
    border: 1px solid #e0e0e0;
    border-radius: 8px;
    margin-bottom: 1rem;
    transition: all 0.3s ease;
    overflow: hidden;
}

.aluno-card:hover {
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    border-color: #d0d0d0;
}

.aluno-card-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 1rem;
    background: #f8f9fa;
    border-bottom: 1px solid #e0e0e0;
    gap: 1rem;
    flex-wrap: wrap;
    cursor: pointer;
    transition: background-color 0.2s ease;
    user-select: none;
}

.aluno-card-header:hover {
    background: #f0f0f0;
}

.aluno-header-right {
    display: flex;
    align-items: center;
    gap: 1rem;
    flex-shrink: 0;
}

.expand-icon {
    color: #666;
    transition: transform 0.3s ease;
    font-size: 14px;
    width: 24px;
    height: 24px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.aluno-card.expanded .expand-icon {
    transform: rotate(180deg);
}

.aluno-avatar {
    flex-shrink: 0;
}

.avatar-circle {
    width: 48px;
    height: 48px;
    border-radius: 50%;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 16px;
}

.aluno-info-main {
    flex: 1;
    min-width: 200px;
}

.aluno-name {
    font-size: 16px;
    margin-bottom: 4px;
}

.aluno-contact {
    font-size: 13px;
    margin-top: 4px;
}

.aluno-actions {
    display: flex;
    gap: 0.5rem;
    flex-shrink: 0;
}

@media (max-width: 768px) {
    .aluno-card-header {
        flex-direction: column;
        align-items: flex-start;
    }
    
    .aluno-header-right {
        width: 100%;
        justify-content: space-between;
    }
    
    .aluno-actions {
        flex-grow: 1;
        justify-content: flex-end;
    }
    
    .aluno-contact {
        flex-direction: column;
        gap: 0.5rem;
    }
    
    .aluno-contact span {
        display: block;
    }
    
    .aluno-contact span.ms-3 {
        margin-left: 0 !important;
    }
}

.aluno-card-body {
    padding: 0;
    max-height: 0;
    overflow: hidden;
    transition: max-height 0.3s ease, padding 0.3s ease;
}

.aluno-card-body.show {
    padding: 1rem;
    max-height: 2000px;
}

.aluno-card.expanded .aluno-card-body {
    padding: 1rem;
    max-height: 2000px;
}

.info-item {
    padding: 0.5rem 0;
}

.info-item small {
    font-size: 11px;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.aluno-row {
    display: block;
}

.aluno-row[data-status="pendente"] {
    border-left: 4px solid #ffc107;
}

.aluno-row[data-status="realizado"] {
    border-left: 4px solid #28a745;
}

</style>

<!-- Modal para detalhes do aluno -->
<div class="modal fade" id="detalhesModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="fas fa-user me-2"></i>Detalhes do Aluno
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="detalhesContent">
                <!-- Conteúdo carregado via JavaScript -->
            </div>
        </div>
    </div>
</div>

<!-- Modal para Novo Pagamento -->
<div class="modal fade" id="novoPagamentoModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-success text-white">
                <h5 class="modal-title">
                    <i class="fas fa-plus-circle me-2"></i>Gerar Novo Link de Pagamento
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="formNovoPagamento">
                    <input type="hidden" id="novoPagAluno" name="aluno_id">
                    <input type="hidden" id="novoPagNegociacao" name="negociacao_id">
                    <input type="hidden" id="novoPagSaldoMax" name="saldo_maximo">
                    
                    <div class="alert alert-info">
                        <strong>Saldo Restante:</strong> R$ <span id="saldoRestanteDisplay">0,00</span>
                    </div>
                    
                    <div class="mb-3">
                        <label for="novoValorPagamento" class="form-label">
                            <i class="fas fa-dollar-sign me-1"></i>Valor do Pagamento: <span class="text-danger">*</span>
                        </label>
                        <div class="input-group">
                            <span class="input-group-text">R$</span>
                            <input type="number" 
                                   class="form-control" 
                                   id="novoValorPagamento" 
                                   name="valor" 
                                   step="0.01" 
                                   min="0.01"
                                   required
                                   oninput="validarValorPagamento()">
                        </div>
                        <small class="text-muted">Valor não pode ultrapassar o saldo restante</small>
                        <div id="erroValor" class="text-danger small mt-1" style="display: none;"></div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="novaFormaPagamento" class="form-label">
                            <i class="fas fa-credit-card me-1"></i>Forma de Pagamento: <span class="text-danger">*</span>
                        </label>
                        <select class="form-select" id="novaFormaPagamento" name="forma_pagamento" required>
                            <option value="pix">PIX</option>
                            <option value="credito">Cartão de Crédito</option>
                        </select>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <i class="fas fa-times me-1"></i>Cancelar
                </button>
                <button type="button" class="btn btn-success" onclick="confirmarNovoPagamento()">
                    <i class="fas fa-link me-1"></i>Gerar Link
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Modal para Exibir Link de Pagamento -->
<div class="modal fade" id="linkPagamentoModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header bg-success text-white">
                <h5 class="modal-title">
                    <i class="fas fa-check-circle me-2"></i>Link de Pagamento Gerado com Sucesso!
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="alert alert-success">
                    <i class="fas fa-info-circle me-2"></i>
                    <strong>Link gerado com sucesso!</strong> O link foi salvo e pode ser visualizado em "Logs Pagar.me"
                </div>
                
                <div class="mb-3">
                    <label for="linkPagamentoInput" class="form-label">
                        <i class="fas fa-link me-1"></i>Link de Pagamento:
                    </label>
                    <div class="input-group">
                        <input type="text" 
                               class="form-control" 
                               id="linkPagamentoInput" 
                               readonly
                               value="">
                        <button class="btn btn-outline-primary" type="button" onclick="copiarLinkPagamento(this)">
                            <i class="fas fa-copy me-1"></i>Copiar
                        </button>
                    </div>
                </div>
                
                <div class="mb-3">
                    <label class="form-label">
                        <i class="fas fa-dollar-sign me-1"></i>Valor do Pagamento:
                    </label>
                    <p class="form-control-plaintext fw-bold text-success" id="valorPagamentoDisplay">R$ 0,00</p>
                </div>
                
                <div class="mb-3">
                    <label class="form-label">
                        <i class="fas fa-wallet me-1"></i>Novo Saldo Restante:
                    </label>
                    <p class="form-control-plaintext" id="novoSaldoDisplay">R$ 0,00</p>
                </div>
                
                <div class="mb-3">
                    <label class="form-label">
                        <i class="fas fa-barcode me-1"></i>Order ID:
                    </label>
                    <p class="form-control-plaintext text-muted small" id="orderIdDisplay">-</p>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <i class="fas fa-times me-1"></i>Fechar
                </button>
                <a href="index.php?page=logs_pagamento" class="btn btn-primary" target="_blank">
                    <i class="fas fa-external-link-alt me-1"></i>Ver em Logs Pagar.me
                </a>
            </div>
        </div>
    </div>
</div>

<?php
// Funções auxiliares para formatação
function formatCPF($cpf) {
    return preg_replace('/(\d{3})(\d{3})(\d{3})(\d{2})/', '$1.$2.$3-$4', $cpf);
}

function formatTelefone($telefone) {
    if (strlen($telefone) == 11) {
        return preg_replace('/(\d{2})(\d{5})(\d{4})/', '($1) $2-$3', $telefone);
    } elseif (strlen($telefone) == 10) {
        return preg_replace('/(\d{2})(\d{4})(\d{4})/', '($1) $2-$3', $telefone);
    }
    return $telefone;
}
?>

<script>
// Toggle card (expandir/recolher)
function toggleCard(header) {
    const card = header.closest('.aluno-card');
    const body = card.querySelector('.aluno-card-body');
    
    // Se já está expandido, recolher
    if (card.classList.contains('expanded')) {
        card.classList.remove('expanded');
        body.classList.remove('show');
        body.style.maxHeight = '0px';
    } else {
        // Expandir
        card.classList.add('expanded');
        body.classList.add('show');
        // Calcular altura real do conteúdo
        const scrollHeight = body.scrollHeight;
        body.style.maxHeight = scrollHeight + 'px';
    }
}

// Filtros agora funcionam via GET params, removida função antiga

// Gerar novo pagamento
function gerarNovoPagamento(alunoId, negociacaoId, saldoRestante) {
    console.log('Gerando novo pagamento:', {alunoId, negociacaoId, saldoRestante});
    
    // Preencher campos do modal
    document.getElementById('novoPagAluno').value = alunoId;
    document.getElementById('novoPagNegociacao').value = negociacaoId;
    document.getElementById('novoPagSaldoMax').value = saldoRestante;
    document.getElementById('saldoRestanteDisplay').textContent = saldoRestante.toFixed(2).replace('.', ',');
    document.getElementById('novoValorPagamento').max = saldoRestante;
    document.getElementById('novoValorPagamento').value = '';
    document.getElementById('erroValor').style.display = 'none';
    
    // Abrir modal
    const modal = new bootstrap.Modal(document.getElementById('novoPagamentoModal'));
    modal.show();
}

// Validar valor do pagamento
function validarValorPagamento() {
    const input = document.getElementById('novoValorPagamento');
    const saldoMax = parseFloat(document.getElementById('novoPagSaldoMax').value);
    const valorDigitado = parseFloat(input.value);
    const erroDiv = document.getElementById('erroValor');
    
    if (valorDigitado > saldoMax) {
        erroDiv.textContent = `⚠️ Valor ultrapassa o saldo restante de R$ ${saldoMax.toFixed(2).replace('.', ',')}`;
        erroDiv.style.display = 'block';
        input.classList.add('is-invalid');
        return false;
    } else if (valorDigitado <= 0) {
        erroDiv.textContent = '⚠️ Valor deve ser maior que zero';
        erroDiv.style.display = 'block';
        input.classList.add('is-invalid');
        return false;
    } else {
        erroDiv.style.display = 'none';
        input.classList.remove('is-invalid');
        return true;
    }
}

// Confirmar novo pagamento
async function confirmarNovoPagamento() {
    console.log('=== confirmarNovoPagamento INICIADO ===');
    
    if (!validarValorPagamento()) {
        console.log('Validação falhou');
        return;
    }
    
    const form = document.getElementById('formNovoPagamento');
    const formData = new FormData(form);
    
    // Converter para objeto
    const data = {};
    formData.forEach((value, key) => {
        data[key] = value;
    });
    
    console.log('Dados a enviar:', data);
    
    // Desabilitar botão
    const btn = event.target;
    const originalText = btn.innerHTML;
    btn.disabled = true;
    btn.innerHTML = '<i class="fas fa-spinner fa-spin me-1"></i>Gerando...';
    
    try {
        console.log('Fazendo requisição para: /portal/especialista/ajax/gerar_novo_pagamento.php');
        
        const response = await fetch('/portal/especialista/ajax/gerar_novo_pagamento.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(data)
        });
        
        console.log('Response status:', response.status);
        
        const responseText = await response.text();
        console.log('Response text:', responseText);
        
        let result;
        try {
            result = JSON.parse(responseText);
        } catch (e) {
            console.error('Erro ao parsear JSON:', e);
            throw new Error('Resposta inválida do servidor: ' + responseText);
        }
        
        console.log('Resultado:', result);
        
        if (result.success) {
            // Fechar modal de novo pagamento
            bootstrap.Modal.getInstance(document.getElementById('novoPagamentoModal')).hide();
            
            // Preencher dados no modal de link gerado
            document.getElementById('linkPagamentoInput').value = result.payment_link;
            document.getElementById('valorPagamentoDisplay').textContent = 
                'R$ ' + parseFloat(result.valor_pagamento || 0).toFixed(2).replace('.', ',');
            document.getElementById('novoSaldoDisplay').textContent = 
                'R$ ' + parseFloat(result.novo_saldo || 0).toFixed(2).replace('.', ',');
            document.getElementById('orderIdDisplay').textContent = result.order_id || '-';
            
            // Mostrar modal de link gerado
            const linkModal = new bootstrap.Modal(document.getElementById('linkPagamentoModal'));
            linkModal.show();
            
            // Recarregar página após fechar o modal
            document.getElementById('linkPagamentoModal').addEventListener('hidden.bs.modal', function () {
                location.reload();
            }, { once: true });
        } else {
            alert('❌ Erro ao gerar link: ' + result.message);
        }
    } catch (error) {
        console.error('ERRO COMPLETO:', error);
        alert('❌ Erro ao gerar link de pagamento: ' + error.message);
    } finally {
        btn.disabled = false;
        btn.innerHTML = originalText;
        }
}

// Mostrar detalhes do aluno
function mostrarDetalhes(alunoId) {
    // Fazer requisição AJAX para buscar detalhes
    fetch(`/portal/includes/get_aluno_detalhes.php?id=${alunoId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                document.getElementById('detalhesContent').innerHTML = data.html;
                new bootstrap.Modal(document.getElementById('detalhesModal')).show();
            } else {
                alert('Erro ao carregar detalhes do aluno.');
            }
        })
        .catch(error => {
            console.error('Erro:', error);
            alert('Erro ao carregar detalhes do aluno.');
        });
}

// Copiar link de pagamento
function copiarLinkPagamento(btnElement) {
    const linkInput = document.getElementById('linkPagamentoInput');
    const linkValue = linkInput.value;
    
    if (!linkValue) {
        alert('Não há link para copiar!');
        return;
    }
    
    // Copiar para área de transferência
    navigator.clipboard.writeText(linkValue).then(function() {
        // Feedback visual
        const btn = btnElement || event.target.closest('button');
        const originalHTML = btn.innerHTML;
        btn.innerHTML = '<i class="fas fa-check me-1"></i>Copiado!';
        btn.classList.remove('btn-outline-primary');
        btn.classList.add('btn-success');
        
        setTimeout(() => {
            btn.innerHTML = originalHTML;
            btn.classList.remove('btn-success');
            btn.classList.add('btn-outline-primary');
        }, 2000);
    }).catch(function(err) {
        // Fallback para navegadores mais antigos
        linkInput.select();
        linkInput.setSelectionRange(0, 99999); // Para mobile
        document.execCommand('copy');
        
        // Feedback visual no fallback
        const btn = btnElement || event.target.closest('button');
        const originalHTML = btn.innerHTML;
        btn.innerHTML = '<i class="fas fa-check me-1"></i>Copiado!';
        btn.classList.remove('btn-outline-primary');
        btn.classList.add('btn-success');
        
        setTimeout(() => {
            btn.innerHTML = originalHTML;
            btn.classList.remove('btn-success');
            btn.classList.add('btn-outline-primary');
        }, 2000);
    });
}



// Atalhos de teclado removidos (filtros via formulário agora)

// Tooltip para atalhos
document.addEventListener('DOMContentLoaded', function() {
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[title]'));
    tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });
});
</script>
