@extends('layouts.coordenador')

@section('title', 'Kanban de Atendimentos')

@section('content')
<div class="content-header">
    <h1 class="content-title">Atendimentos - Minhas Equipes</h1>
    <p class="content-subtitle">Visualize os leads das suas equipes e em qual etapa cada especialista os posicionou</p>
</div>

<!-- Estatísticas -->
<div class="stats-grid">
    <!-- Total de Leads -->
    <div class="stat-card">
        <div class="stat-header">
            <div class="stat-icon" style="background: #dbeafe; color: #3b82f6;">
                <i class="fas fa-users"></i>
            </div>
        </div>
        <div class="stat-value">{{ number_format($stats->total ?? 0, 0, ',', '.') }}</div>
        <div class="stat-label">Total de Leads</div>
    </div>

    <!-- Aguardando -->
    <div class="stat-card">
        <div class="stat-header">
            <div class="stat-icon" style="background: #fef3c7; color: #f59e0b;">
                <i class="fas fa-clock"></i>
            </div>
        </div>
        <div class="stat-value">{{ number_format($stats->aguardando ?? 0, 0, ',', '.') }}</div>
        <div class="stat-label">Aguardando</div>
    </div>

    <!-- Em Negociação -->
    <div class="stat-card">
        <div class="stat-header">
            <div class="stat-icon" style="background: #dcfce7; color: #22c55e;">
                <i class="fas fa-handshake"></i>
            </div>
        </div>
        <div class="stat-value">{{ number_format($stats->negociacao ?? 0, 0, ',', '.') }}</div>
        <div class="stat-label">Em Negociação</div>
    </div>

    <!-- Aprovados -->
    <div class="stat-card">
        <div class="stat-header">
            <div class="stat-icon" style="background: #dcfce7; color: #22c55e;">
                <i class="fas fa-check-circle"></i>
            </div>
        </div>
        <div class="stat-value">{{ number_format($stats->aprovada ?? 0, 0, ',', '.') }}</div>
        <div class="stat-label">Aprovados</div>
    </div>
</div>

<!-- Filtros de Pesquisa -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" action="{{ route('coordenador.atendimentos.index') }}" id="filtrosForm">
            <div class="row g-3 align-items-end">
                <!-- Busca por Nome/Email/Telefone -->
                <div class="col-md-5">
                    <label class="form-label text-muted small mb-1">Buscar</label>
                    <input type="text" 
                           name="busca" 
                           class="form-control" 
                           placeholder="Nome, email ou telefone..."
                           value="{{ $filtroBusca }}">
                </div>
                
                <!-- Filtro por Atribuição -->
                <div class="col-md-5">
                    <label class="form-label text-muted small mb-1">Responsável</label>
                    <select name="atribuido" class="form-select">
                        <option value="">Todos</option>
                        <option value="nao_atribuido" {{ $filtroAtribuido === 'nao_atribuido' ? 'selected' : '' }}>
                            Não Atribuídos
                        </option>
                        <optgroup label="Especialistas">
                            @foreach($especialistas as $esp)
                                <option value="esp_{{ $esp->id }}" {{ $filtroAtribuido === "esp_{$esp->id}" ? 'selected' : '' }}>
                                    {{ $esp->nome }}
                                </option>
                            @endforeach
                        </optgroup>
                    </select>
                </div>
                
                <!-- Botões -->
                <div class="col-md-2">
                    <div class="d-flex gap-2">
                        <button type="submit" class="btn btn-primary flex-fill">
                            <i class="fas fa-search"></i>
                        </button>
                        @if(!empty($filtroBusca) || !empty($filtroAtribuido))
                            <a href="{{ route('coordenador.atendimentos.index') }}" class="btn btn-outline-secondary" title="Limpar filtros">
                                <i class="fas fa-times"></i>
                            </a>
                        @endif
                    </div>
                </div>
            </div>
            
            @if(!empty($filtroBusca) || !empty($filtroAtribuido))
                <div class="mt-3 pt-3 border-top">
                    <small class="text-muted">
                        <i class="fas fa-filter me-1"></i>
                        {{ $leads->count() }} resultado(s) encontrado(s)
                    </small>
                </div>
            @endif
        </form>
    </div>
</div>

<!-- Kanban Board -->
<div class="card">
    <div class="card-header bg-white">
        <h5 class="mb-0">
            <i class="fas fa-columns me-2"></i>
            Kanban Board - Gestão de Atendimentos
        </h5>
    </div>
    <div class="card-body">
        <div class="kanban-container">
            <div class="kanban-board" id="kanban-board">
                <!-- Colunas serão geradas via JavaScript -->
            </div>
        </div>
    </div>
</div>

<!-- Modal para Detalhes do Lead -->
<div class="details-modal" id="detailsModal">
    <div class="modal-content">
        <div class="modal-header">
            <h3 class="modal-title">
                <i class="fas fa-info-circle me-2"></i>Detalhes do Lead
            </h3>
            <button class="modal-close" onclick="closeDetailsModal()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <div class="modal-body" id="detailsContent">
            <!-- Conteúdo será inserido via JavaScript -->
        </div>
    </div>
</div>

@push('styles')
<style>
/* Estilos específicos para o Kanban do Coordenador */
.kanban-container {
    width: 100%;
    overflow-x: auto;
    padding: 20px 0;
}

.kanban-board {
    display: flex;
    gap: 20px;
    min-width: fit-content;
}

.kanban-column {
    flex: 0 0 300px;
    background: #f8f9fa;
    border-radius: 12px;
    padding: 16px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    display: flex;
    flex-direction: column;
    min-height: 500px;
}

.kanban-column.drag-over {
    background: rgba(34, 197, 94, 0.05);
    border: 2px dashed #22c55e;
}

.kanban-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 16px;
    padding-bottom: 12px;
    border-bottom: 2px solid rgba(0,0,0,0.1);
}

.kanban-title {
    display: flex;
    align-items: center;
    gap: 8px;
    font-weight: 600;
    font-size: 14px;
    color: #333;
}

.kanban-title i {
    font-size: 18px;
}

.kanban-count {
    background: white;
    padding: 4px 12px;
    border-radius: 20px;
    font-weight: 600;
    font-size: 14px;
}

.kanban-cards {
    display: flex;
    flex-direction: column;
    gap: 12px;
    min-height: 400px;
    flex: 1;
}

.kanban-card {
    background: white;
    border-radius: 8px;
    padding: 16px;
    box-shadow: 0 2px 6px rgba(0,0,0,0.1);
    cursor: pointer;
    transition: all 0.2s;
    position: relative;
}

.kanban-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
}

.kanban-card.lead-distribuido {
    border-left: 4px solid #22c55e;
}

.kanban-card[draggable="true"] {
    cursor: move;
}

.kanban-card.dragging {
    opacity: 0.5;
    transform: rotate(3deg) scale(0.95);
    box-shadow: 0 8px 20px rgba(0, 0, 0, 0.3);
}

.kanban-column.drag-over .kanban-cards {
    background: rgba(34, 197, 94, 0.15);
    border-radius: 8px;
}

.kanban-card[draggable="true"]::before {
    content: '\f0c9';
    font-family: 'Font Awesome 6 Free';
    font-weight: 900;
    position: absolute;
    left: 8px;
    top: 50%;
    transform: translateY(-50%);
    color: #ddd;
    font-size: 12px;
    opacity: 0;
    transition: opacity 0.2s;
}

.kanban-card[draggable="true"]:hover::before {
    opacity: 1;
}

.negociacao-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 8px;
}

.negociacao-id {
    font-size: 12px;
    color: #666;
    font-weight: 600;
}

.negociacao-title {
    font-size: 16px;
    font-weight: 600;
    margin: 8px 0;
    color: #333;
}

.negociacao-meta {
    display: flex;
    flex-direction: column;
    gap: 6px;
    margin-top: 12px;
}

.meta-item {
    display: flex;
    align-items: center;
    gap: 8px;
    font-size: 13px;
    color: #666;
}

.meta-item i {
    font-size: 14px;
    width: 16px;
}

.whatsapp-link {
    color: #25D366;
    text-decoration: none;
    display: flex;
    align-items: center;
    gap: 6px;
    transition: color 0.2s;
}

.whatsapp-link:hover {
    color: #128C7E;
}

.especialista-tag {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    background: #dcfce7;
    color: #16a34a;
    padding: 4px 10px;
    border-radius: 12px;
    font-size: 11px;
    font-weight: 600;
    margin-top: 8px;
}

.view-details-btn {
    position: absolute;
    top: 12px;
    right: 12px;
    background: white;
    border: 1px solid #ddd;
    border-radius: 6px;
    width: 32px;
    height: 32px;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all 0.2s;
    z-index: 10;
}

.view-details-btn:hover {
    background: #f0f0f0;
    border-color: #999;
}

.kanban-empty {
    text-align: center;
    padding: 40px 20px;
    color: #999;
}

.kanban-empty i {
    font-size: 48px;
    margin-bottom: 12px;
    opacity: 0.3;
}

.details-modal {
    display: none;
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0,0,0,0.5);
    z-index: 9999;
    overflow-y: auto;
}

.details-modal.show {
    display: flex;
    align-items: center;
    justify-content: center;
}

.details-modal .modal-content {
    background: white;
    border-radius: 12px;
    width: 90%;
    max-width: 600px;
    max-height: 90vh;
    overflow-y: auto;
    box-shadow: 0 4px 20px rgba(0,0,0,0.3);
}

.details-modal .modal-header {
    padding: 20px;
    border-bottom: 1px solid #dee2e6;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.details-modal .modal-title {
    font-size: 20px;
    font-weight: 600;
    margin: 0;
}

.details-modal .modal-close {
    background: none;
    border: none;
    font-size: 24px;
    cursor: pointer;
    color: #666;
    padding: 0;
    width: 32px;
    height: 32px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 4px;
    transition: all 0.2s;
}

.details-modal .modal-close:hover {
    background: #f0f0f0;
    color: #333;
}

.details-modal .modal-body {
    padding: 20px;
}
</style>
@endpush

@push('scripts')
<script>
// Dados dos leads vindos do Laravel
const LEADS_DATA = {!! json_encode($leads) !!};

console.log('═══════════════════════════════════════════════');
console.log('🔍 COORDENADOR - Atendimentos - Minhas Equipes');
console.log('═══════════════════════════════════════════════');
console.log('Total de leads:', LEADS_DATA.length);

// Configuração das colunas do kanban
const kanbanConfig = {
    'aguardando': {
        label: 'Aguardando Contato',
        icon: 'fas fa-clock',
        color: 'linear-gradient(90deg, #ff9800, #ffc107)'
    },
    'contatado': {
        label: 'Contatado',
        icon: 'fas fa-phone',
        color: 'linear-gradient(90deg, #2196f3, #1e88e5)'
    },
    'negociacao': {
        label: 'Em Negociação',
        icon: 'fas fa-handshake',
        color: 'linear-gradient(90deg, #22c55e, #16a34a)'
    },
    'aprovada': {
        label: 'Aprovada',
        icon: 'fas fa-check-circle',
        color: 'linear-gradient(90deg, #4caf50, #43a047)'
    },
    'rejeitada': {
        label: 'Perdida',
        icon: 'fas fa-times-circle',
        color: 'linear-gradient(90deg, #f44336, #e53935)'
    }
};

// Organizar leads por status
function organizeLeadsByStatus() {
    const organized = {
        'aguardando': [],
        'contatado': [],
        'negociacao': [],
        'aprovada': [],
        'rejeitada': []
    };
    
    LEADS_DATA.forEach(lead => {
        // Usar status_kanban se disponível, senão mapear de etapa
        let targetStatus = lead.status_kanban || 'aguardando';
        
        // Se não tiver status_kanban, mapear de etapa
        if (!lead.status_kanban && lead.etapa) {
            const etapaMap = {
                'novo': 'aguardando',
                'contato': 'contatado',
                'demo': 'negociacao',
                'fechamento': 'aprovada',
                'perdido': 'rejeitada'
            };
            targetStatus = etapaMap[lead.etapa] || 'aguardando';
        }
        
        // Garantir que o status existe
        if (!organized[targetStatus]) {
            targetStatus = 'aguardando';
        }
        
        organized[targetStatus].push(lead);
    });
    
    console.log('📊 Leads organizados por status:', organized);
    return organized;
}

// Estado para drag and drop
let draggedCard = null;

// Criar card de lead
function createLeadCard(lead) {
    const card = document.createElement('div');
    card.className = 'kanban-card';
    card.dataset.id = `lead-${lead.id}`;
    card.dataset.leadId = lead.id;
    
    // Determinar status do kanban
    let statusKanban = lead.status_kanban || 'aguardando';
    if (!lead.status_kanban && lead.etapa) {
        const etapaMap = {
            'novo': 'aguardando',
            'contato': 'contatado',
            'demo': 'negociacao',
            'fechamento': 'aprovada',
            'perdido': 'rejeitada'
        };
        statusKanban = etapaMap[lead.etapa] || 'aguardando';
    }
    
    card.dataset.status = statusKanban;
    
    // Coordenador pode arrastar apenas leads em "aguardando" ou "contatado"
    if (statusKanban === 'aguardando' || statusKanban === 'contatado') {
        card.draggable = true;
        card.addEventListener('dragstart', handleDragStart);
        card.addEventListener('dragend', handleDragEnd);
    } else {
        card.style.cursor = 'default';
    }
    
    // Adicionar classe se está distribuído
    if (lead.distribuido) {
        card.classList.add('lead-distribuido');
    }
    
    // Formatar telefone para WhatsApp
    let whatsappLink = '#';
    let telefoneDisplay = lead.telefone || 'Sem telefone';
    
    if (lead.telefone) {
        const telefoneNumeros = lead.telefone.replace(/\D/g, '');
        if (telefoneNumeros.length >= 10) {
            whatsappLink = `https://wa.me/55${telefoneNumeros}`;
        }
    }
    
    // Formatar data
    let dataExibicao = 'Sem data';
    if (lead.horario_preferencia) {
        dataExibicao = lead.horario_preferencia;
    } else if (lead.data_cadastro) {
        const data = new Date(lead.data_cadastro);
        dataExibicao = data.toLocaleDateString('pt-BR');
    }
    
    const isDraggable = (statusKanban === 'aguardando' || statusKanban === 'contatado');
    
    card.innerHTML = `
        <div class="negociacao-header">
            <span class="negociacao-id">#${String(lead.id).padStart(4, '0')}</span>
            ${isDraggable ? '<i class="fas fa-grip-vertical text-muted" style="font-size: 12px;" title="Arraste para mover"></i>' : ''}
        </div>
        
        <button class="view-details-btn" 
                onclick="event.stopPropagation(); openDetailsModal(${lead.id})" 
                draggable="false"
                title="Ver detalhes completos">
            <i class="fas fa-eye"></i>
        </button>
        
        <h4 class="negociacao-title">${escapeHtml(lead.nome || 'Lead sem nome')}</h4>
        
        <div class="negociacao-meta">
            <div class="meta-item">
                <a href="${whatsappLink}" 
                   target="_blank" 
                   class="whatsapp-link" 
                   draggable="false"
                   onclick="event.stopPropagation()"
                   title="Abrir WhatsApp">
                    <i class="fab fa-whatsapp"></i>
                    <span>${escapeHtml(telefoneDisplay)}</span>
                </a>
            </div>
            <div class="meta-item">
                <i class="fas fa-calendar"></i>
                <span>${dataExibicao}</span>
            </div>
            ${lead.especialista_nome ? `
            <div class="especialista-tag">
                <i class="fas fa-user-tie"></i>
                ${escapeHtml(lead.especialista_nome)}
            </div>
            ` : lead.equipe_nome ? `
            <div class="especialista-tag" style="background: #fef3c7; color: #f59e0b;">
                <i class="fas fa-users"></i>
                ${escapeHtml(lead.equipe_nome)}
            </div>
            ` : `
            <div class="especialista-tag" style="background: #fee2e2; color: #dc2626;">
                <i class="fas fa-exclamation-triangle"></i>
                Não atribuído
            </div>
            `}
        </div>
    `;
    
    return card;
}

// Renderizar o kanban board
function renderKanbanBoard() {
    console.log('🎨 Renderizando Kanban Board Coordenador...');
    
    const board = document.getElementById('kanban-board');
    if (!board) {
        console.error('❌ Elemento #kanban-board não encontrado!');
        return;
    }
    
    const organizedData = organizeLeadsByStatus();
    board.innerHTML = '';
    
    Object.entries(kanbanConfig).forEach(([status, config]) => {
        const leads = organizedData[status] || [];
        
        // Criar coluna
        const column = document.createElement('div');
        column.className = 'kanban-column';
        column.dataset.status = status;
        column.style.setProperty('--column-color', config.color);
        
        // Header da coluna
        const header = document.createElement('div');
        header.className = 'kanban-header';
        header.innerHTML = `
            <div class="kanban-title">
                <i class="${config.icon}"></i>
                <span>${config.label}</span>
            </div>
            <div class="kanban-count">${leads.length}</div>
        `;
        
        // Container dos cards
        const cardsContainer = document.createElement('div');
        cardsContainer.className = 'kanban-cards';
        
        if (leads.length === 0) {
            cardsContainer.innerHTML = `
                <div class="kanban-empty">
                    <i class="${config.icon}"></i>
                    <p>Nenhum lead</p>
                </div>
            `;
        } else {
            leads.forEach(lead => {
                const card = createLeadCard(lead);
                cardsContainer.appendChild(card);
            });
        }
        
        column.appendChild(header);
        column.appendChild(cardsContainer);
        board.appendChild(column);
        
        // Adicionar eventos de drop na COLUNA INTEIRA para aguardando e contatado
        if (status === 'aguardando' || status === 'contatado') {
            column.addEventListener('dragover', handleDragOver);
            column.addEventListener('dragleave', handleDragLeave);
            column.addEventListener('drop', handleDrop);
        }
    });
}

// Handlers para drag and drop
function handleDragStart(e) {
    const target = e.target;
    
    if (target.closest('a') || target.closest('button') || target.tagName === 'A' || target.tagName === 'BUTTON') {
        e.preventDefault();
        return false;
    }
    
    const card = target.closest('.kanban-card');
    if (!card) {
        e.preventDefault();
        return false;
    }
    
    draggedCard = card;
    card.classList.add('dragging');
    e.dataTransfer.effectAllowed = 'move';
    e.dataTransfer.setData('text/html', card.outerHTML);
}

function handleDragEnd(e) {
    const card = e.target.closest('.kanban-card') || draggedCard;
    if (card) {
        card.classList.remove('dragging');
    }
    draggedCard = null;
}

function handleDragOver(e) {
    e.preventDefault();
    e.dataTransfer.dropEffect = 'move';
    
    const targetColumn = e.target.closest('.kanban-column');
    if (targetColumn && draggedCard) {
        const currentStatus = draggedCard.dataset.status;
        const targetStatus = targetColumn.dataset.status;
        
        if (['aguardando', 'contatado'].includes(currentStatus) && 
            ['aguardando', 'contatado'].includes(targetStatus)) {
            targetColumn.classList.add('drag-over');
        }
    }
}

function handleDragLeave(e) {
    const targetColumn = e.target.closest('.kanban-column');
    if (targetColumn) {
        targetColumn.classList.remove('drag-over');
    }
}

function handleDrop(e) {
    e.preventDefault();
    
    const targetColumn = e.target.closest('.kanban-column');
    
    if (targetColumn) {
        targetColumn.classList.remove('drag-over');
    }
    
    if (!draggedCard) return;
    
    const targetCards = targetColumn ? targetColumn.querySelector('.kanban-cards') : null;
    
    if (targetColumn && targetCards) {
        const currentCardsContainer = draggedCard.parentElement;
        
        if (currentCardsContainer === targetCards) {
            return;
        }
        
        const newStatus = targetColumn.dataset.status;
        const leadId = draggedCard.dataset.leadId;
        const oldStatus = draggedCard.dataset.status;
        
        const statusPermitidos = ['aguardando', 'contatado'];
        
        if (!statusPermitidos.includes(oldStatus) || !statusPermitidos.includes(newStatus)) {
            showErrorMessage('Você só pode mover leads entre "Aguardando Contato" e "Contatado"');
            return;
        }
        
        // Mover o card para a nova coluna
        targetCards.appendChild(draggedCard);
        draggedCard.dataset.status = newStatus;
        
        // Atualizar interface imediatamente
        updateColumnCounts();
        updateEmptyMessages();
        
        // Atualizar no servidor
        atualizarStatusLead(leadId, newStatus, oldStatus);
    }
}

function escapeHtml(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Abrir modal de detalhes
function openDetailsModal(leadId) {
    const lead = LEADS_DATA.find(l => l.id == leadId);
    
    if (!lead) {
        alert('Lead não encontrado');
        return;
    }
    
    const modal = document.getElementById('detailsModal');
    const content = document.getElementById('detailsContent');
    
    const telefoneNumeros = (lead.telefone || '').replace(/\D/g, '');
    const whatsappLink = telefoneNumeros.length >= 10 ? `https://wa.me/55${telefoneNumeros}` : '#';
    
    content.innerHTML = `
        <div class="row mb-3">
            <div class="col-md-6">
                <strong>ID:</strong><br>
                <span class="text-muted">#${lead.id}</span>
            </div>
            <div class="col-md-6">
                <strong>Status:</strong><br>
                <span class="badge badge-primary">${escapeHtml(lead.status)}</span>
            </div>
        </div>
        
        <div class="row mb-3">
            <div class="col-md-12">
                <strong>Nome:</strong><br>
                <span class="text-muted">${escapeHtml(lead.nome)}</span>
            </div>
        </div>
        
        <div class="row mb-3">
            <div class="col-md-6">
                <strong>Email:</strong><br>
                <span class="text-muted">${escapeHtml(lead.email)}</span>
            </div>
            <div class="col-md-6">
                <strong>Telefone:</strong><br>
                <a href="${whatsappLink}" target="_blank" class="text-success">
                    <i class="fab fa-whatsapp me-1"></i>${escapeHtml(lead.telefone)}
                </a>
            </div>
        </div>
        
        ${lead.profissao ? `
        <div class="row mb-3">
            <div class="col-md-12">
                <strong>Profissão:</strong><br>
                <span class="text-muted">${escapeHtml(lead.profissao)}</span>
            </div>
        </div>
        ` : ''}
        
        ${lead.especialista_nome ? `
        <div class="row mb-3">
            <div class="col-md-12">
                <strong>Especialista Responsável:</strong><br>
                <span class="badge badge-success">
                    <i class="fas fa-user-tie me-1"></i>${escapeHtml(lead.especialista_nome)}
                </span>
            </div>
        </div>
        ` : lead.equipe_nome ? `
        <div class="row mb-3">
            <div class="col-md-12">
                <strong>Equipe Responsável:</strong><br>
                <span class="badge badge-info">
                    <i class="fas fa-users me-1"></i>${escapeHtml(lead.equipe_nome)}
                </span>
            </div>
        </div>
        ` : `
        <div class="alert alert-warning">
            <i class="fas fa-exclamation-triangle me-2"></i>
            Lead ainda não foi atribuído a nenhum especialista ou equipe
        </div>
        `}
        
        <div class="row mb-3">
            <div class="col-md-6">
                <strong>Data de Cadastro:</strong><br>
                <span class="text-muted">${lead.data_cadastro ? new Date(lead.data_cadastro).toLocaleDateString('pt-BR') : 'Não informado'}</span>
            </div>
            ${lead.data_distribuicao ? `
            <div class="col-md-6">
                <strong>Data de Distribuição:</strong><br>
                <span class="text-muted">${new Date(lead.data_distribuicao).toLocaleDateString('pt-BR')}</span>
            </div>
            ` : ''}
        </div>
        
        ${lead.observacoes ? `
        <div class="mb-3">
            <strong>Observações:</strong><br>
            <div class="border rounded p-3 bg-light">
                ${escapeHtml(lead.observacoes)}
            </div>
        </div>
        ` : ''}
    `;
    
    modal.classList.add('show');
}

function closeDetailsModal() {
    const modal = document.getElementById('detailsModal');
    modal.classList.remove('show');
}

// Fechar modal clicando fora
document.addEventListener('click', function(e) {
    const modal = document.getElementById('detailsModal');
    if (e.target === modal) {
        closeDetailsModal();
    }
});

// Fechar modal com ESC
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeDetailsModal();
    }
});

// Atualizar contadores das colunas
function updateColumnCounts() {
    document.querySelectorAll('.kanban-column').forEach(column => {
        const cardsContainer = column.querySelector('.kanban-cards');
        const visibleCards = cardsContainer ? cardsContainer.querySelectorAll('.kanban-card').length : 0;
        const countElement = column.querySelector('.kanban-count');
        if (countElement) {
            countElement.textContent = visibleCards;
        }
    });
}

// Atualizar mensagens de coluna vazia
function updateEmptyMessages() {
    document.querySelectorAll('.kanban-column').forEach(column => {
        const cardsContainer = column.querySelector('.kanban-cards');
        if (!cardsContainer) return;
        
        const cards = cardsContainer.querySelectorAll('.kanban-card');
        const emptyMessage = cardsContainer.querySelector('.kanban-empty');
        const status = column.dataset.status;
        const config = kanbanConfig[status];
        
        if (cards.length === 0 && !emptyMessage) {
            // Adicionar mensagem de vazio
            const emptyDiv = document.createElement('div');
            emptyDiv.className = 'kanban-empty';
            emptyDiv.innerHTML = `
                <i class="${config.icon}"></i>
                <p>Nenhum lead</p>
            `;
            cardsContainer.appendChild(emptyDiv);
        } else if (cards.length > 0 && emptyMessage) {
            // Remover mensagem de vazio
            emptyMessage.remove();
        }
    });
}

// Atualizar status do lead no banco de dados
async function atualizarStatusLead(leadId, novoStatus, statusAnterior) {
    try {
        const response = await fetch(`{{ url('/coordenador/atendimentos') }}/${leadId}/status`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
            },
            body: JSON.stringify({
                status: novoStatus
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            const statusLabel = novoStatus === 'aguardando' ? 'Aguardando Contato' : 'Contatado';
            showSuccessMessage(`✅ Lead movido para "${statusLabel}"`);
            
            // Atualizar contadores e mensagens de coluna vazia
            updateColumnCounts();
            updateEmptyMessages();
        } else {
            showErrorMessage('❌ Erro: ' + result.message);
            
            // Reverter movimento
            const originalColumn = document.querySelector(`[data-status="${statusAnterior}"] .kanban-cards`);
            if (originalColumn && draggedCard) {
                originalColumn.appendChild(draggedCard);
                draggedCard.dataset.status = statusAnterior;
                updateColumnCounts();
                updateEmptyMessages();
            }
        }
    } catch (error) {
        console.error('Erro ao atualizar status:', error);
        showErrorMessage('❌ Erro ao atualizar status');
        
        // Reverter movimento
        const originalColumn = document.querySelector(`[data-status="${statusAnterior}"] .kanban-cards`);
        if (originalColumn && draggedCard) {
            originalColumn.appendChild(draggedCard);
            draggedCard.dataset.status = statusAnterior;
            updateColumnCounts();
        }
    }
}

// Mostrar mensagem de sucesso
function showSuccessMessage(message) {
    const alert = document.createElement('div');
    alert.className = 'alert alert-success alert-dismissible fade show position-fixed';
    alert.style.cssText = 'top: 80px; right: 20px; z-index: 9999; min-width: 300px;';
    alert.innerHTML = `
        <i class="fas fa-check-circle me-2"></i>
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.body.appendChild(alert);
    
    setTimeout(() => {
        alert.remove();
    }, 4000);
}

// Mostrar mensagem de erro
function showErrorMessage(message) {
    const alert = document.createElement('div');
    alert.className = 'alert alert-danger alert-dismissible fade show position-fixed';
    alert.style.cssText = 'top: 80px; right: 20px; z-index: 9999; min-width: 300px;';
    alert.innerHTML = `
        <i class="fas fa-exclamation-triangle me-2"></i>
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.body.appendChild(alert);
    
    setTimeout(() => {
        alert.remove();
    }, 4000);
}

// Inicializar quando a página carregar
document.addEventListener('DOMContentLoaded', function() {
    renderKanbanBoard();
});
</script>
@endpush

@endsection

